/*==========================================================================================
 This program replicates the main results for the RDP on Anticipatory Monetary Policy and 
 the 'Price Puzzle' by James Bishop and Peter Tulip. 

 It reads in two datasets from Excel. The first uses forecasts from 'forecast-month Board 
 meetings' and the second using Board meetings. 
**========================================================================================*/

clear all
version 12
pause on
set more off
set maxvar 15000

ssc install estout 

* replace the directory of the folder

local directory "G:\Research\JamesB\Price puzzle"

cd "`directory'\replication_files (public release)"

/*==========================================================================================
  Import data and generate key variables
**========================================================================================*/

import excel using data.xlsx, sheet("STATA_q") cellrange(A1:AQ138) firstrow

gen qtr=yq(year,quarter)
format qtr %tq

* declare data to be time series

sort qtr
tsset qtr

* Variable transformations

gen ln_cpi_trim = ln(cpi_trim)
gen infl        = ln_cpi_trim-L1.ln_cpi_trim

foreach x in rgdp usgdp comm rtwi {
gen ln_`x'     = ln(`x')
gen dln_`x'    = ln_`x'-L1.ln_`x'
}
*
gen cr_qtr_ch = cr_qtr-L1.cr_qtr

cd "`directory'\replication_files (public release)\stata_files"

save qtr_shock, replace

/*==========================================================================================
  Construct Figure 1
  
  Figure 1 in the RDP presents the IRFs from a 6-variable recursive VAR. It imposes block-exogeneity of the 
  foreign variables with respect to the domestic variables, with standard errors generated using a parametric
  bootstrap (1,000 draws). 
  
  The relevant impulse responses, along with 90% CIs are exported to the STATA output file figure1.xls. Excel 
  versions of Figure 1 are drawn in: \\san1\er\Research\JamesB\Price puzzle\Results.xlsx (which links to the
  STATA output file). 
  
**========================================================================================*/

* Impose restrictions on lagged effects, reflecting block exogeneity of foreign with respect to domestic variables

constraint 1 [ln_usgdp]L1.ln_rgdp = 0
constraint 2 [ln_usgdp]L1.infl    = 0
constraint 3 [ln_usgdp]L1.ln_rtwi = 0
constraint 4 [ln_comm]L1.ln_rgdp  = 0
constraint 5 [ln_comm]L1.infl     = 0
constraint 6 [ln_comm]L1.ln_rtwi  = 0
constraint 7 [ln_usgdp]L1.cr_qtr  = 0
constraint 8 [ln_comm]L1.cr_qtr   = 0

constraint 9  [ln_usgdp]L2.ln_rgdp= 0
constraint 10 [ln_usgdp]L2.infl   = 0
constraint 11 [ln_usgdp]L2.ln_rtwi= 0
constraint 12 [ln_comm]L2.ln_rgdp = 0
constraint 13 [ln_comm]L2.infl    = 0
constraint 14 [ln_comm]L2.ln_rtwi = 0
constraint 15 [ln_usgdp]L2.cr_qtr = 0
constraint 16 [ln_comm]L2.cr_qtr  = 0

constraint 17 [ln_usgdp]L3.ln_rgdp= 0
constraint 18 [ln_usgdp]L3.infl   = 0
constraint 19 [ln_usgdp]L3.ln_rtwi= 0
constraint 20 [ln_comm]L3.ln_rgdp = 0
constraint 21 [ln_comm]L3.infl    = 0
constraint 22 [ln_comm]L3.ln_rtwi = 0
constraint 23 [ln_usgdp]L3.cr_qtr = 0
constraint 24 [ln_comm]L3.cr_qtr  = 0

constraint 25 [infl]L1.cr_qtr= 0
constraint 26 [infl]L2.cr_qtr= 0
constraint 27 [infl]L3.cr_qtr= 0


* estimate recursive VAR

var ln_usgdp ln_comm ln_rgdp infl cr_qtr ln_rtwi if qtr>=tq(1990q2),  lags (1 2 3) constraints(1/24)

irf create order1, step(30) set(impulse_fig1) bsp reps(1000) replace 

use "`directory'\replication_files (public release)\stata_files\impulse_fig1.irf", clear

sort impulse response step
keep step oirf coirf stdoirf stdcoirf irfname response impulse
keep if impulse=="cr_qtr"
keep if response=="infl" | response=="ln_rgdp" | response=="cr_qtr" 

* Scale IRFs to be for a 1ppt positive shock to the cash rate
sum oirf if step==0 & impulse=="cr_qtr" & response=="cr_qtr"
scalar shock_size = r(mean)

foreach x in oirf coirf stdoirf stdcoirf {
replace `x'=(`x'/shock_size) 
replace `x'=`x'*100 if response=="infl" | response=="ln_rgdp" 
}
* 
* generate 90 per cent CIs
gen u90o_ = oirf +(1.645*stdoirf)
gen l90o_ = oirf -(1.645*stdoirf)

gen u90c_ = coirf+(1.645*stdcoirf)
gen l90c_ = coirf-(1.645*stdcoirf)

drop stdoirf stdcoirf
drop if response=="cr_qtr"
rename oirf  oirf_
rename coirf coirf_

reshape wide oirf_ coirf_ u90o_ l90o_ u90c_ l90c_, i(step) j(response) string
keep step oirf_ln_rgdp u90o_ln_rgdp l90o_ln_rgdp oirf_infl u90o_infl l90o_infl coirf_infl u90c_infl l90c_infl
order step oirf_ln_rgdp u90o_ln_rgdp l90o_ln_rgdp oirf_infl u90o_infl l90o_infl coirf_infl u90c_infl l90c_infl

export excel using figure1, firstrow(variables) replace
clear


/*==========================================================================================
  Construct Figure 2
  
  Figure 2 in the RDP decomposes the price puzzle in A VAR (see explanation in RDP). Since the VAR 
  is recursive, an easy way to implement this is to estimate the VAR equation-by-equation, and  
  then simulate a cash rate shock both with and without certain coefficients being set equal to
  zero, after estimation. These simulations are performed in the 'Decompositions' tab in the following:
  
  \\san1\er\Research\JamesB\Price puzzle\Results.xlsx
  
  The above spreadhseet links directly to a STATA output file (recursive.csv), generated below.
**========================================================================================*/

cd "`directory'\replication_files (public release)\stata_files"

use qtr_shock, clear

* Estimate recursive VAR equation-by-equation
* to run this you may need to install the package 'estout' To do this, type 'ssc install estout' in the command line
eststo:  reg ln_usgdp L(1/3).(ln_comm)    L(1/3).ln_usgdp if qtr>=tq(1990q2)
eststo:  reg ln_comm  L(0/3).(ln_usgdp)   L(1/3).ln_comm  if qtr>=tq(1990q2)
eststo:  reg ln_rgdp  L(0/3).(ln_usgdp ln_comm)                     L(1/3).(ln_rgdp infl cr_qtr ln_rtwi) if qtr>=tq(1990q2)
eststo:  reg infl     L(0/3).(ln_usgdp ln_comm ln_rgdp)             L(1/3).(infl cr_qtr ln_rtwi) if qtr>=tq(1990q2)
eststo:  reg cr_qtr   L(0/3).(ln_usgdp ln_comm ln_rgdp infl)        L(1/3).(cr_qtr ln_rtwi) if qtr>=tq(1990q2)
eststo:  reg ln_rtwi  L(0/3).(ln_usgdp ln_comm ln_rgdp infl cr_qtr) L(1/3).ln_rtwi if qtr>=tq(1990q2)

esttab using recursive.csv, nostar not plain replace


/*==========================================================================================
 Estimate first-stage regressions (part i)
 
 Our baseline approach uses cash rate changes and forecasts from 'forecast-month Board meetings', with
 Newey-West standard errors. 
 
 The first 5 alternative specifications (see Section 3 of RDP) are also based on 'forecast-month Board 
 meetings'. 
**========================================================================================*/


**===== Baseline

newey cr_change cr_prev yegdpf_2 yeinff_2 yegdpf_2r yeinff_2r unef_0 if qtr>=tq(1991q1), lag(1) force 


* obtain monetary policy shocks as the residuals

predict mpshock if e(sample), residuals

* joint tests of significance

test (yegdpf_2 yegdpf_2r)
test (yeinff_2 yeinff_2r)
test (_b[yegdpf_2] + _b[yegdpf_2r]=0)
test (_b[yeinff_2] + _b[yeinff_2r]=0)


**===== Robustness to Alternative Specifications (#1-#5)

* Approach 1: extending forecasts out to 4-quarters ahead
*   If 4-qtr-ahead forecast (or forecast revision) not available, use 3-qtr-ahead or, failing that, 2-qtr-ahead

foreach x in yegdpf_ yeinff_  {
gen `x'4i = `x'4
replace `x'4i = `x'3 if `x'4i==.
}
foreach x in yegdpf_ yeinff_  {
gen `x'4ri = `x'4r
replace `x'4ri = `x'3r if `x'4ri==.
replace `x'4ri = `x'2r if `x'4ri==.
}
newey cr_change cr_prev yegdpf_0 yegdpf_4i yeinff_0 yeinff_4i yegdpf_4ri yeinff_4ri unef_0 if qtr>=tq(1991q1), lag(1) force 
predict mpshock_4per if e(sample), residuals

* Approach 2: replace GDP forecasts with forecasts for unemployment rate
newey cr_change cr_prev unef_2 yeinff_2 unef_2r yeinff_2r unef_0 if qtr>=tq(1991q1), lag(1) force 
predict mpshock_une2 if e(sample), residuals
newey cr_change cr_prev unef_4 yeinff_4 unef_4r yeinff_4r unef_0 if qtr>=tq(1991q1), lag(1) force 
predict mpshock_une4 if e(sample), residuals

* Approach 3: include cash rate change from previous meeting (NB: previous month, not previous forecast-meeting month)
newey cr_change cr_prev yegdpf_2 yeinff_2 yegdpf_2r yeinff_2r unef_0 cr_change_l1 if qtr>=tq(1991q1), lag(1) force 
predict mpshock_chg if e(sample), residuals

* Approach 4: post-1996 sample
newey cr_change cr_prev yegdpf_2 yeinff_2 yegdpf_2r yeinff_2r unef_0 if qtr>=tq(1996q1), lag(1) force 
predict mpshock_96 if e(sample), residuals

* Approach 5(a): Cochrane specification for inflation
newey cr_change yeinff_2 yeinff_2r if qtr>=tq(1991q1), lag(1) force 
predict mpshock_inf2 if e(sample), residuals
newey cr_change yeinff_4 yeinff_4r if qtr>=tq(1991q1), lag(1) force 
predict mpshock_inf4 if e(sample), residuals

* Approach 5(b): Cochrane specification for GDP
newey cr_change yegdpf_2 yegdpf_2r if qtr>=tq(1991q1), lag(1) force 
predict mpshock_gdp2 if e(sample), residuals
newey cr_change yegdpf_4 yegdpf_4r if qtr>=tq(1991q1), lag(1) force 
predict mpshock_gdp4 if e(sample), residuals


**===== Treatment of missing data

* for observations with missing forecast data, we omit these observations from the first-stage and set the shocks equal to zero
foreach x in mpshock mpshock_4per mpshock_une2 mpshock_une4 mpshock_inf2 mpshock_inf4 mpshock_gdp2 mpshock_gdp4 mpshock_chg mpshock_96 {
replace `x'=0 if `x'==. & qtr>=tq(1991q2) & qtr<=tq(2016q1)
}
*
replace mpshock_96=. if qtr<=tq(1995q4)


**========== Extract the orthonalised (structural) shocks from a recursive VAR (Approach 12).

* Easiest approach to constructing Choleski shock is just to estimate the structural cash rate equation directly, and obtain residuals.
* These results were cross-checked using a more sophisticated approach; see end of do file. 

reg cr_qtr   L(0/3).(ln_usgdp ln_comm ln_rgdp infl)  L(1/3).(cr_qtr ln_rtwi) if qtr>=tq(1991q2)
predict mpshock_chol if e(sample), residuals

cd "`directory'\replication_files (public release)\stata_files"
save qtr_shock, replace
clear



/*==========================================================================================
 Estimate first-stage regressions (part ii)
 
 The remaining alternative specifications use information from all Board meethings, rather than just
 'forecast-month' meetings
 
**========================================================================================*/

cd "`directory'\replication_files (public release)"

* import data
import excel using data.xlsx, sheet("STATA_m") cellrange(A1:M377) firstrow
* note: the paper imports a larger file with confidential data filling columns N to BD

* create variable denoting quarter 
gen mth=ym(year,month)
format mth %tm

* declare data to be time series
sort mth
tsset mth

**========== Alternative specifications

* Approach 6: use most recent offical forecasts
* also accounts for differences in JEFG/SMP timing for inflation and GDP forecast
newey cr_change cr_prev yegdpf_2 yeinff_2 yegdpf_2r yeinff_2r unef_0 if mth>=tm(1991m1), lag(1) force 
predict mpshock_mth1, residuals

/*  NOTE: the following code is commented out given that the data cannot be released to the public

* Approach 7: using PDG/Board forecasts (2001 onwards)
newey cr_change cr_prev yegdpf_2pdg yeinff_2pdg yegdpf_2pdgr yeinff_2pdgr unef_0pdg, lag(1) force 
predict mpshock_pdg if e(sample), residuals

* Approach 8: Consensus forecasts
replace cons_nex_gdpr=0 if cons_nex_gdpr==.
replace cons_nex_cpir=0 if cons_nex_cpir==.
newey cr_change cr_prev cons_cur_gdp cons_nex_gdp cons_cur_gdpr cons_nex_gdpr cons_cur_cpi cons_nex_cpi cons_cur_cpir cons_nex_cpir i.month if month!=1, lag(1) force 
predict mpshock_cons if e(sample), residuals

* Approach 9: post-1985 sample using BRW survey 
* set missing 'next' year forecasts to zero and define a missing data dummy

gen     f_miss = 0
replace f_miss=1 if brw_nex_gdp==.
replace brw_nex_cpi = 0 if f_miss==1
replace brw_nex_gdp = 0 if f_miss==1
newey cr_change cr_prev brw_cur_gdp brw_nex_gdp brw_cur_gdpr brw_cur_cpi brw_nex_cpi brw_cur_cpir f_miss i.month pub_lag if month!=1, lag(1) force 
predict mpshock_brw if e(sample), residuals

*/

* Aggregate shocks to qtly frequency by summing across all meetings in the quarter (with non-missing data)
* NOTE: this look usually includes "mth2 cons brw pdg"

foreach x in mpshock_mth1 km_sup_1m km_sup_3m {
replace `x'=0 if month==1
}
*
collapse (sum) mpshock_mth1 km_sup_1m km_sup_3m, by(year quarter)

gen q = yq(year, quarter)
format q %tq

replace mpshock_mth1=. if q<=tq(1991q1)
replace km_sup_1m=. if q<=tq(1989q4)
replace km_sup_3m=. if q<=tq(1989q4)

drop q

cd "`directory'\replication_files (public release)\stata_files"
sort year quarter
save mth_shock, replace
clear


/*==========================================================================================
  Merge shocks based on quarterly and monthly shocks into a single file 
**========================================================================================*/

cd "`directory'\replication_files (public release)\stata_files"

use qtr_shock, clear
sort year quarter
merge 1:1 year quarter using mth_shock
drop _merge

sort qtr
tsset qtr
 
drop yegdpf_4i yeinff_4i yegdpf_4ri yeinff_4ri

* Cumulate shock series for use in VARs

sum    cr_new if qtr>=tq(1991q2)
scalar cr_new_mean = r(mean)

foreach x in mpshock mpshock_4per mpshock_une2 mpshock_une4 mpshock_inf2 mpshock_inf4 mpshock_gdp2 ///
            mpshock_gdp4 mpshock_chg mpshock_chol mpshock_mth1 ///
			km_sup_1m km_sup_3m  {
			
gen     `x'_c = `x'
replace `x'_c=0 if qtr==tq(1991q1)
replace `x'_c =  `x'_c[_n-1] + `x' in 2/l if qtr>=tq(1991q2)
replace `x'_c=. if qtr<tq(1991q1)		

* adjust the cumulated series for the average cash rate over the sample period
sum    `x'_c if qtr>=tq(1991q2)
scalar `x'_c_mean  = r(mean)
replace `x'_c=`x'_c - `x'_c_mean + cr_new_mean			
}
*		

* Post 1996
gen     mpshock_96_c = mpshock_96
replace mpshock_96_c=0 if qtr==tq(1995q4)
replace mpshock_96_c =  mpshock_96_c[_n-1] + mpshock_96 in 2/l if qtr>=tq(1996q1)
replace mpshock_96_c=. if qtr<tq(1995q4)

sum    cr_new if qtr>=tq(1996q1)
scalar cr_new_mean96 = r(mean)
sum    mpshock_96_c if qtr>=tq(1996q1)
scalar mpshock_96_c_mean  = r(mean)
replace mpshock_96_c=mpshock_96_c - mpshock_96_c_mean + cr_new_mean96	



save master, replace			 

			 
/*==========================================================================================
  Single equation models
**========================================================================================*/

foreach p in infl dln_rgdp {

use "`directory'\replication_files (public release)\stata_files\master.dta", clear

var `p'      if qtr>=tq(1991q2) & L12.mpshock!=.,  lags (1(1)8) exog(L(1/12).mpshock)
irf create mpshock, step(30) set(cdm_`p') replace

foreach x in cr_qtr_ch cr_change mpshock_4per mpshock_une2 mpshock_une4 mpshock_inf2 mpshock_inf4 ///
             mpshock_gdp2 mpshock_gdp4 mpshock_chg mpshock_chol mpshock_mth1 ///
			 km_sup_1m km_sup_3m {

quietly var `p'  if qtr>=tq(1991q2) & L12.mpshock!=.,  lags (1(1)8) exog(L(1/12).`x') 
irf create `x', step(30) set(cdm_`p') replace 

}
* post-1996 subsample uses a shorter sample period 
quietly var `p'  if qtr>=tq(1996q1),  lags (1(1)8) exog(L(1/12).mpshock_96) 
irf create mpshock_96, step(30) set(cdm_`p')  replace 

use "`directory'\replication_files (public release)\stata_files\cdm_`p'.irf", clear

drop if impulse=="`p'"
keep step cdm stdcdm irfname
rename irfname impulse

* convert to percentages
replace cdm    =cdm  *100
replace stdcdm=stdcdm*100

gen u90_ = cdm+(1.645*stdcdm)
gen l90_ = cdm-(1.645*stdcdm)
drop stdcdm
rename cdm cdm_

reshape wide cdm_ u90_ l90_, i(step) j(impulse) string
order step *mpshock *cr_qtr_ch *cr_change *mpshock_4per *mpshock_une2 *mpshock_une4 *mpshock_inf2 *mpshock_inf4 *mpshock_gdp2 *mpshock_gdp4 *mpshock_chg *mpshock_chol *mpshock_mth1 *km_sup_1m *km_sup_3m *mpshock_96
export excel using dynamic_multipliers, firstrow(variables) sheet("`p'") sheetreplace
}
*

/*==========================================================================================
  3-Variable VAR
**========================================================================================*/
local lag 3

use "`directory'\replication_files (public release)\stata_files\master.dta", clear
local i


foreach x in mpshock cr_qtr_ch cr_change mpshock_4per mpshock_une2 mpshock_une4 mpshock_inf2 mpshock_inf4 ///
             mpshock_gdp2 mpshock_gdp4 mpshock_chg mpshock_chol mpshock_mth1 ///
			 km_sup_1m km_sup_3m {

quietly var ln_rgdp infl `x'  if qtr>=tq(1991q2) & L3.mpshock!=.,  lags (1/`lag')
irf create `x', step(30) set(irfs) bsp reps(1000) order(ln_rgdp infl `x') replace 

}
*

* post-1996 subsample uses a shorter sample period 
quietly var ln_rgdp infl mpshock_96  if qtr>=tq(1996q1),  lags (1/`lag')
irf create mpshock_96, step(30)  set(irfs) bsp reps(1000) order(ln_rgdp infl mpshock_96) replace 

use "`directory'\replication_files (public release)\stata_files\irfs.irf", clear

drop if impulse=="ln_rgdp" | impulse=="infl"

keep step coirf stdcoirf irfname response
rename irfname impulse

* scale IRFs to be a 1ppt shock to cash rate, and convert to percentages
gen scaling = .
replace scaling = coirf if step==0 & response!=("ln_rgdp") & response!=("infl")
bysort impulse: egen scaling2 = max(scaling)
drop scaling
drop if response!=("ln_rgdp") & response!=("infl")

replace coirf   =(coirf   /scaling2)  *100
replace stdcoirf=(stdcoirf/scaling2)*100

gen u90_ = coirf+(1.645*stdcoirf)
gen l90_ = coirf-(1.645*stdcoirf)
drop stdcoirf
rename coirf coirf_

drop scaling2
sort impulse response step

save irfs_gdp, replace  
drop if response!=("infl")
drop response
save irfs_inf, replace  
use irfs_gdp, clear   
drop if response!=("ln_rgdp")
drop response
save irfs_gdp, replace  

reshape wide coirf_ u90_ l90_, i(step) j(impulse) string
order step *mpshock *cr_qtr_ch *cr_change *mpshock_4per *mpshock_une2 *mpshock_une4 *mpshock_inf2 *mpshock_inf4 *mpshock_gdp2 *mpshock_gdp4 *mpshock_chg *mpshock_chol *mpshock_mth1 *km_sup_1m *km_sup_3m *mpshock_96 
export excel using irfs, firstrow(variables) sheet("gdp") sheetreplace

use irfs_inf, clear
reshape wide coirf_ u90_ l90_, i(step) j(impulse) string
order step *mpshock *cr_qtr_ch *cr_change *mpshock_4per *mpshock_une2 *mpshock_une4 *mpshock_inf2 *mpshock_inf4 *mpshock_gdp2 *mpshock_gdp4 *mpshock_chg *mpshock_chol *mpshock_mth1 *km_sup_1m *km_sup_3m *mpshock_96 
export excel using irfs, firstrow(variables) sheet("infl") sheetreplace


* end of do file





/*==========================================================================================
More detailed approach to calculating the structural innovations in a recursive VAR 
**========================================================================================

constraint 1  [ln_usgdp]L1.ln_rgdp = 0
constraint 2  [ln_usgdp]L1.infl = 0
constraint 3  [ln_usgdp]L1.ln_rtwi = 0
constraint 4  [ln_comm]L1.ln_rgdp = 0
constraint 5  [ln_comm]L1.infl = 0
constraint 6  [ln_comm]L1.ln_rtwi = 0
constraint 7  [ln_usgdp]L2.ln_rgdp = 0
constraint 8  [ln_usgdp]L2.infl = 0
constraint 9  [ln_usgdp]L2.ln_rtwi = 0
constraint 10 [ln_comm]L2.ln_rgdp = 0
constraint 11 [ln_comm]L2.infl = 0
constraint 12 [ln_comm]L2.ln_rtwi = 0
constraint 13 [ln_usgdp]L3.ln_rgdp = 0
constraint 14 [ln_usgdp]L3.infl = 0
constraint 15 [ln_usgdp]L3.ln_rtwi = 0
constraint 16 [ln_comm]L3.ln_rgdp = 0
constraint 17 [ln_comm]L3.infl = 0
constraint 18 [ln_comm]L3.ln_rtwi = 0
constraint 19 [ln_usgdp]L1.cr_qtr = 0
constraint 20 [ln_comm]L1.cr_qtr = 0
constraint 21 [ln_usgdp]L2.cr_qtr = 0
constraint 22 [ln_comm]L2.cr_qtr = 0
constraint 23 [ln_usgdp]L3.cr_qtr = 0
constraint 24 [ln_comm]L3.cr_qtr = 0

var ln_usgdp ln_comm ln_rgdp infl cr_qtr ln_rtwi if qtr>=tq(1991q2),  lags (1 2 3) constraints(1/24) 
matrix  sig_var  = e(Sigma)
matrix  chol_var = cholesky(sig_var)
matlist chol_var

matrix  p_inv = inv(chol_var)
matlist p_inv

predict ln_usgdp_res, res equation(ln_usgdp)
predict ln_comm_res , res equation(ln_comm)
predict ln_rgdp_res , res equation(ln_rgdp)
predict infl_res    , res equation(infl)
predict cr_qtr_res  , res equation(cr_qtr)
predict ln_rtwi_res , res equation(ln_rtwi)

mkmat ln_usgdp_res ln_comm_res ln_rgdp_res infl_res cr_qtr_res ln_rtwi_res , matrix(matrix_res) 

matrix matrix_res_trans=matrix_res'

matrix structuralshocks_trans=p_inv*matrix_res_trans
matrix structuralshocks      =structuralshocks_trans'

svmat  structuralshocks
drop   structuralshocks1 structuralshocks2 structuralshocks3 structuralshocks4 structuralshocks6
rename structuralshocks5 mpshock_chol_cr_qtr 

drop ln_usgdp_res ln_comm_res ln_rgdp_res infl_res ln_rtwi_res

* unlike the reduced form residuals, the B matrix scales the structural innovations to have a unit variance (thus mean zero, SD 1). To compare to the reduced form errors, re-express reduced form errors to have unit variance 
sum cr_qtr_res if qtr>=tq(1991q2) & qtr<=tq(2015q4)
scalar mn1 = r(mean)
scalar sd1 = r(sd)
sum mpshock_chol_cr_qtr if qtr>=tq(1991q2) & qtr<=tq(2015q4)
scalar mn2 = r(mean)
scalar sd2 = r(sd)

gen mpshock_chol_cr_qtr_r  = mn1+(mpshock_chol_cr_qtr-mn2)*(sd1/sd2)

replace mpshock_chol_cr_qtr_r=. if qtr<tq(1991q2) | qtr>tq(2015q4)

rename mpshock_chol_cr_qtr_r        mpshock_chol1

*/ 
