function [llk, AcP, BcP, AX, BX, K0Q_cP, K1Q_cP, rho0_cP, rho1_cP, rho0_cP_r, rho1_cP_r, cP, yields_filtered, cP_filtered] = ...
    jszLLK_KF_r_start(yields_o, yields_n, yields_r, W, K1Q_X, kinfQ, Sigma_cP, rho0_X_r, rho1_X_r, mats, mats_r, dt, K0P_cP, K1P_cP, sigma_e_n, sigma_e_r, s_infl, s_infl_hor, s_infl_std, s_inflQ, s_inflQ_hor, s_inflQ_std,s_inflLR, s_inflLR_hor, s_inflLR_std, s_cash, s_cash_hor,s_cash_std,find_r)

% This function computes the likelihood for a Gaussian term structure model via a Kalaman filter, as well as the fitted values.
% See "Affine Endeavour: Estimating a Joint Model of the Real and Nominal Term
% Structures in Australia" by Jonathan Hambur and Richard Finlay (2018)
% Uses Kalman filter without the perfect pricing assumption
% Adapated from code in JSZ
%
% INPUTS: - 
% yields_o   : T*J,      matrix of observed yields (first row are t=0 observations, which likelihood conditions on)
% mats       : 1*J_n,    Maturities in years for nominal  
% mats       : 1*J_n,    Maturities in years for real
% dt         : scalar,   length of period in years (1/12 is months)
% W          : N*J,      vector of portfolio weights to fit without error.
% K1Q_X      : N*N,      normalized latent-model matrix (Q dynamics)
% kinfQ      : scalar,   when the model is stationary, the long run mean of the annualized short rate under Q is -kinfQ/K1(m1,m1) 
% Sigma_cP   : N*N,      positive definite matrix that is the covariance of innovations to cP
% K0P_cP     : N*1,      Intercept for P-dynamics
% K1P_cP     : N*N,      Tranistion matrix for P-dynamics
% sigma_e    : scalar,   standard error of yield observation errors
% s_""       :           Matrices of survey observations
% s_""_hor   :           Vectors of survey horizons (in years)
%
%
% Compute likelihood conditioned on first observation!
%
% llk        : T*1       time series of -log likelihoods (includes 2-pi constants)
% AcP        : 1*J       yt = AcP' + BcP'*cPt  (yt is J*1 vector)
% BcP        : N*J       AcP, BcP satisfy internal consistency condition that AcP*W' = 0, BcP*W' = I_N
% AX         : 1*J       yt = AX' + BX'*Xt  
% BX         : N*J       Xt is the 'jordan-normalized' latent state
% yields_filtered : T*J  E[cPt|y^o(t), y^o(t-1), .., y^o(1)]
% cP_filtered     : T*N  E[yt|y^o(t), y^o(t-1), .., y^o(1)]
%
% The model takes the form:
%   r(t) = rho0_cP + rho1_cP'*cPt
%        = rinfQ + 1'*Xt  (Xt is the 'jordan-normalized' state
%        = 1 period discount rate (annualized)
%
%   infl(t) = rho0_cP_r + rho1_cP_r'*cPt
%           = rho0_X_r + rho1_X_r'*cPt 
%Where the mapping beteween the  two sets of coeffcient on short-rates is
%the same. See Appendix B 

% Under Q:
%   X(t+1) - X(t)   =          K1Q_X*X(t)  + eps_X(t+1),   cov(eps_X(t+1)) = Sigma_X
%   cP(t+1) - cP(t) = K0Q_cP + K1Q_cP*X(t) + eps_cP(t+1),  cov(eps_cP(t+1)) = Sigma_cP
%   where Sigma_X is chosen to match Sigma_cP 
%
% Under P:
%   cP(t+1) - cP(t) = K0P_cP + K1P_cP*X(t) + eps_cP(t+1),  cov(eps_cP(t+1)) = Sigma_cP
%
%

    
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Setup 
[T,J] = size(yields_o);
[T,J_n] = size(yields_n);
[T,~] = size(yields_r);
% Setup W if we are using individual yields without error:
if isempty(W)
    W = eye(J);
    W = W(ismember(mats, yields_woe),:); % N*J    
end
N = size(W,1);
cP = yields_o*W'; % (T+1)*N, cP stands for math caligraphic P.

% Check if specified K0P_cP or whether want to constrain to have SS of cP
% as mean of cP (small sample adjustment to mean)

concentrateAlpha = true; %calculate K0P_cP or implied by average and K1P_cP 
if exist('K0P_cP','var') && ~isempty(K0P_cP), concentrateAlpha = false; end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Convert from JSZ-Normalized version of the model:
% yt = AcP' + BcP'*cPt, AcP is 1*J, BcP is N*J
[BcP, AcP, K0Q_cP, K1Q_cP, rho0_cP, rho1_cP, rho0_cP_r, rho1_cP_r, K0Q_X, K1Q_X, AX, BX, Sigma_X] = jszLoadings_r(W, K1Q_X, kinfQ, Sigma_cP, rho0_X_r, rho1_X_r, mats, mats_r, dt);
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Now setup Kalman filter.
y_n = yields_n;  % T*J
y_r = yields_r;  % T*J
y_cash=log(1.+s_cash); % Turning surveys into continuous compounding as in model
y_infl=log(1.+s_infl);
y_inflQ=log(1.+s_inflQ);
y_inflLR=log(1.+s_inflLR);

Phi = eye(N) + K1P_cP; % N*N Transition matrix

%If making small sample adjsutment, set K0P_cP to ensure mean of cP is
%steady state. If not set to calibrated level
if concentrateAlpha==true
    alpha= mean(cP)'-(K1P_cP+eye(N))*mean(cP)';
else
    alpha = K0P_cP; % N*1
end

Q = Sigma_cP;  % N*N Covariance matrix

%Variances for the surveys and yield errors - treats as homoskedastic
R_n = sigma_e_n^2;
R_r = sigma_e_r^2;
R_cash=s_cash_std.^2*ones(1,size(s_cash,2)); 
R_infl=s_infl_std.^2*ones(1,size(s_infl,2));
R_inflQ=s_inflQ_std.^2*ones(1,size(s_inflQ,2));
R_inflLR=s_inflLR_std.^2*ones(1,size(s_inflLR,2));

% Assume that the t=0 states (the time before any yields are observed)
% are distributed N(mu, Sigma) with the model stationary distribution,
[x00, P00] = asymptoticMomentsGaussian(alpha, K1P_cP, Sigma_cP);
x00 = cP(1,:).'; %Rather than start at asympttic mean, start at initial point

% If K1P_cP is non-stationary, we will have a problem with this assumption,
% so use something else for covariance
eigP00 = eig(P00);
if any(~isreal(eigP00)) || any(eigP00<0)
    x00 = cP(1,:).';
    P00 = cov(cP);
end

%Seperate out nominal and real yield coeffcieints
A_n = BcP(:,1:J_n).';
b_n = AcP(:,1:J_n).';
A_r = BcP(:,J_n+1:end).';
b_r = AcP(:,J_n+1:end).';


%Get coefficients to fit surveys (i.e. fitted survey=b+A*cP)

% First, get coefficients required to iterate forward expectations by
% required number of periods, basd on P-dynamics.
[b_cash, A_cash] = fit_short_coeff(rho0_cP, rho1_cP, alpha,K1P_cP, round(s_cash_hor*12),N); %short-rates
[b_inflation, A_inflation] = fit_short_coeff(rho0_cP_r, rho1_cP_r, alpha,K1P_cP, 1:160,N); %Infaltion

% For short-rates, this is sufficeint, as just forecasting cash rate in x-
% months. For inflation now need to convert into relevant suvey measures
% (e.g. 5-10 year average etc)
A_inflation=A_inflation./12; % de-annualise
b_inflation=b_inflation./12;
[b_infl, A_infl] = infl_direct_coeff(b_inflation, A_inflation, round(s_infl_hor*12), N);
[b_inflQ, A_inflQ] = infl_q_coeff(b_inflation, A_inflation, round(s_inflQ_hor*12), N);
[b_inflLR, A_inflLR]= infl_LR_coeff(b_inflation, A_inflation, round(s_inflLR_hor*12), N);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Run Kalman Filter:
[x_tm1t, P_tm1t, x_tt, P_tt, K_t, llk] = kf_r( y_n.',y_r.', y_cash.',y_infl.', y_inflQ.',y_inflLR.', Phi, alpha, A_n, b_n,A_r, b_r, A_cash.', b_cash.', A_infl.', b_infl.', A_inflQ.', b_inflQ.', A_inflLR.', b_inflLR.', Q, R_n, R_r, R_cash, R_infl, R_inflQ, R_inflLR, x00, P00, find_r);
llk = -llk; % we return negative of the llk
cP_filtered = x_tt.';     % T*N  E[cPt|y^o(t), y^o(t-1), .., y^o(1)]
yields_filtered = cP_filtered*BcP + ones(T,1)*AcP; % T*J
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
