/**************************************************************
  Program created by James Bishop during RBA inposting in 2016. It uses
  the WPI micro data to estimate the effect of award wages on the probability
  of job loss.  
  **************************************************************/

clear all

set more off

cd "\\sasnasprd\SASData\W386\RBA\dta_files\master"

use transitions, clear

/**************************************************************
 For each decision there is a pre-treatment transition and a post-treatment
 transition.
 
 When a job ceases, it drops out of the panel. 
 
 For computational simplicity, measure the pre-treatment (Nov-May) transition as 
 at Nov (d=0) and the post-treatment (May-Nov) transition as at May (d=1). Define 
 wage groups as at the start of each transition. 
 
 For each transitional period, the sample includes all workers who were alive at
 the start of the transition, with a valid wage observation.
**************************************************************/

/**************************************************************
* Split sample into different pre- and post-experiments
**************************************************************/

gen     treatment_no = .

* WRA:
replace treatment_no = 1  if surveyq==tq(1997q4) | surveyq==tq(1998q2)
replace treatment_no = 2  if surveyq==tq(1998q4) | surveyq==tq(1999q2)
replace treatment_no = 3  if surveyq==tq(1999q4) | surveyq==tq(2000q2)
replace treatment_no = 4  if surveyq==tq(2000q4) | surveyq==tq(2001q2)
replace treatment_no = 5  if surveyq==tq(2001q4) | surveyq==tq(2002q2)
replace treatment_no = 6  if surveyq==tq(2002q4) | surveyq==tq(2003q2)
replace treatment_no = 7  if surveyq==tq(2003q4) | surveyq==tq(2004q2)
replace treatment_no = 8  if surveyq==tq(2004q4) | surveyq==tq(2005q2)
* WorkChoices
replace treatment_no = 9  if surveyq==tq(2006q2) | surveyq==tq(2006q4)
replace treatment_no = 10 if surveyq==tq(2007q1) | surveyq==tq(2007q3)
replace treatment_no = 11 if surveyq==tq(2008q1) | surveyq==tq(2008q3)
* Minimum wage Freeze:
replace treatment_no = 12 if surveyq==tq(2009q1) | surveyq==tq(2009q3)
* Modernisation, flat dollar increase:
replace treatment_no = 13 if surveyq==tq(2009q4) | surveyq==tq(2010q2)
* Modernisation, flat percentage increase:
replace treatment_no = 14 if surveyq==tq(2010q4) | surveyq==tq(2011q2)
replace treatment_no = 15 if surveyq==tq(2011q4) | surveyq==tq(2012q2)
replace treatment_no = 16 if surveyq==tq(2012q4) | surveyq==tq(2013q2)
replace treatment_no = 17 if surveyq==tq(2013q4) | surveyq==tq(2014q2)
replace treatment_no = 18 if surveyq==tq(2014q4) | surveyq==tq(2015q2)


* Treatment period dummy
gen     d = .
replace d = 0 if j_quarter==4 & (treatment_no>=1  & treatment_no<=8) 
replace d = 1 if j_quarter==2 & (treatment_no>=1  & treatment_no<=8)

replace d = 0 if surveyq==tq(2006q2)
replace d = 0 if surveyq==tq(2007q1)
replace d = 0 if surveyq==tq(2008q1)
replace d = 0 if surveyq==tq(2009q1)

replace d = 1 if surveyq==tq(2006q4)
replace d = 1 if surveyq==tq(2007q3)
replace d = 1 if surveyq==tq(2008q3)
replace d = 1 if surveyq==tq(2009q3)

replace d = 0 if j_quarter==4 & (treatment_no>=14  & treatment_no<=18) 
replace d = 1 if j_quarter==2 & (treatment_no>=14  & treatment_no<=18)

* Define treatment group (awd=1) as individuals paid award wage AT START OF Pre-TRANSITION or post-TRANSITION
gen     awd = .
replace awd = 0        if j_eba_ddd==1 & treatment_no!=. & (d==0 | d==1)
replace awd = 1        if j_awardm2==1 & treatment_no!=. & (d==0 | d==1)

* quarter dummies
tab surveyq if treatment_no!=., gen(qtr_) 


/**************************************************************
* SNR Adjustments
**************************************************************/

forvalues c = 1(1)2 {

gen     snr`c' = .

replace snr`c' = 14 if week_awd`c' <  550                     & treatment_no==1 & week_awd`c'!=.
replace snr`c' = 12 if week_awd`c' >= 550 & week_awd`c' < 700 & treatment_no==1 & week_awd`c'!=. 
replace snr`c' = 10 if week_awd`c' >= 700                     & treatment_no==1 & week_awd`c'!=. 

replace snr`c' = 12 if week_awd`c' <  510                     & treatment_no==2 & week_awd`c'!=.
replace snr`c' = 10 if week_awd`c' >= 510                     & treatment_no==2 & week_awd`c'!=.

replace snr`c' = 15 if                                          treatment_no==3 & week_awd`c'!=. 

replace snr`c' = 13 if week_awd`c' <  490                     & treatment_no==4 & week_awd`c'!=. 
replace snr`c' = 15 if week_awd`c' >= 490 & week_awd`c' < 590 & treatment_no==4 & week_awd`c'!=. 
replace snr`c' = 17 if week_awd`c' >= 590                     & treatment_no==4 & week_awd`c'!=. 

replace snr`c' = 18 if                                          treatment_no==5 & week_awd`c'!=. 

replace snr`c' = 17 if week_awd`c' <  731.80                  & treatment_no==6 & week_awd`c'!=. 
replace snr`c' = 15 if week_awd`c' >= 731.80                  & treatment_no==6 & week_awd`c'!=. 

replace snr`c' = 19 if                                          treatment_no==7 & week_awd`c'!=.

replace snr`c' = 17 if                                          treatment_no==8 & week_awd`c'!=. 

replace snr`c' = 27.36 if week_awd`c' <  700                  & treatment_no==9 & week_awd`c'!=.
replace snr`c' = 22.04 if week_awd`c' >= 700                  & treatment_no==9 & week_awd`c'!=. 

replace snr`c' = 10.26 if week_awd`c' <  700                  & treatment_no==10 & week_awd`c'!=. 
replace snr`c' =  5.30 if week_awd`c' >= 700                  & treatment_no==10 & week_awd`c'!=. 

replace snr`c' = 21.66 if                                       treatment_no==11 & week_awd`c'!=. 

* The SNR for 2009 is the amount that we assume for the placebo test. Assume the same as the 2008 decision
replace snr`c' = 21.66     if                                   treatment_no==12 & week_awd`c'!=. 

replace snr`c' = 26.10 if                                       treatment_no==13 & week_awd`c'!=. 

* Placebo (2015)
replace snr`c' = 16.00 if                                       treatment_no==18 & week_awd`c'!=.


/**************************************************************
 Award adjustment
**************************************************************/

* Predicted % increase in base wage under SNR

gen     week_awd_adj`c' = week_awd`c'
replace week_awd_adj`c' = fmw       if week_awd`c' < fmw & (week_awd`c'!=. & fmw!=.) 

sort    j_id surveyq

gen     snr_adj`c'  = ln(week_awd_adj`c'+snr`c') - ln(week_awd_adj`c')

}
sort  j_id surveyq




/**************************************************************
* Job cessation

  Simplest approach is to use all cases where defnilqt==C. 
  
  In some cases, a job is recorded as 'temporarily dead' (defnilqt==T) at t but then 
  dead at t+1. In this case, assign the time of death to period t. After recoding a cessation
  of the job, it is no longer in the sample.
  
  In many cases, this variable also captures firm failure. But a few other cases of firm failure
  can be captured using information from the firm-level file. Variable for quarter of firm death 
  was constructed in clean_mueaa902.do

  Also include cases where the entire firm fails.
 
**************************************************************/

gen m_defnilqt_c= 1 if m_defnilqt=="C"
gen m_defnilqt_t= 1 if m_defnilqt=="T"

gen     job_cease = 0
replace job_cease = 1 if m_defnilqt_c==1 & L1.m_defnilqt_t!=1
replace job_cease = 1 if m_defnilqt_t==1 & F1.m_defnilqt_c==1 & L1.m_defnilqt_t
replace job_cease = 1 if m_defnilqt_t==1 & F1.m_defnilqt_t==1 & F2.m_defnilqt_c==1

* dummy = 1 if job alive at end of transition (conditional on being alive at start of transition)
* note: for simplicity, measured from the start of the transition
gen      survive1 = .
replace  survive1 = 1 if job_cease==0 &  F1.job_cease==0 & F2.job_cease==0  & d!=.
replace  survive1 = 0 if job_cease==0 & (F1.job_cease==1 | F2.job_cease==1) & d!=.

gen    e_death_qtrl1 = e_death_qtr-1
gen    e_death_qtrl2 = e_death_qtr-2
format e_death_qtrl1 %tq
format e_death_qtrl2 %tq

gen      survive2 = survive1
replace  survive2 = 0 if job_cease==0 & (surveyq==e_death_qtrl1 | surveyq==e_death_qtrl2) & d!=.

tab survive1
tab survive2

tab surveyq survive2 if j_awardm2==1

gen     death1 = 0 if survive1==1
replace death1 = 1 if survive1==0
gen     death2 = 0 if survive2==1
replace death2 = 1 if survive2==0


* Variable capturing whether there is also a change in employee in the position (broader measure)

gen      change_e = .
replace  change_e = 1 if  F1.j_same_emp==0 & F2.j_same_emp==0  & d!=.
replace  change_e = 0 if  F1.j_same_emp==1 | F2.j_same_emp==1  & d!=.

gen     death_broad = 0 if                d!=.
replace death_broad = 1 if death2==1    & d!=.
replace death_broad = 1 if change_e==1  & d!=.


* Drop public sector employees
keep if e_public==0

gen     dln_wage = F1.w_wage_spl1_dln + F2.w_wage_spl1_dln if d!=.

gen     dln_hrs = ln(F2.m_ordhrswk_jb) - ln(m_ordhrswk_jb) if d!=.

/**************************************************************
* Using wage decline dummies 

  The 'control decile' is zero in the pre- and post-treatment period. The 'treatment deciles'
  are zero in the pre-treatment period and 1 in the post-treatment period.
**************************************************************/
sort  j_id surveyq


* Calculate deciles of award wage distribution, by year, separately for the pre- and post-transitions 
* Deciles are defined based on award wage at start of each transition 
forvalues s=10(10)90 {
bysort treatment_no d: egen p`s' = pctile(week_awd_adj2) if awd==1 & treatment_no!=., p(`s')

* also assign these deciles to EBA jobs
bysort treatment_no d: egen p`s'a = max(p`s') if treatment_no!=.
drop p`s'
rename p`s'a  p`s'
}
*

* Calculate wage decile categorical variable (1-10) 
* Include EBA jobs, but using the deciles of the award wage distribution rather than EBA distribution 
* Note that not including the 'F2_ln_wage!=.' condition, because want to do this analysis using unbalanced panels

sort  j_id surveyq

gen dec=.

* & dln_wage!=. & dln_hrs!=.
replace dec = 1 if week_awd_adj2< p10                      & week_awd_adj2!=.  & treatment_no!=. 
replace dec = 2 if week_awd_adj2>=p10 & week_awd_adj2< p20 & week_awd_adj2!=.  & treatment_no!=. 
replace dec = 3 if week_awd_adj2>=p20 & week_awd_adj2< p30 & week_awd_adj2!=.  & treatment_no!=. 
replace dec = 4 if week_awd_adj2>=p30 & week_awd_adj2< p40 & week_awd_adj2!=.  & treatment_no!=. 
replace dec = 5 if week_awd_adj2>=p40 & week_awd_adj2< p50 & week_awd_adj2!=.  & treatment_no!=. 
replace dec = 6 if week_awd_adj2>=p50 & week_awd_adj2< p60 & week_awd_adj2!=.  & treatment_no!=. 
replace dec = 7 if week_awd_adj2>=p60 & week_awd_adj2< p70 & week_awd_adj2!=.  & treatment_no!=. 
replace dec = 8 if week_awd_adj2>=p70 & week_awd_adj2< p80 & week_awd_adj2!=.  & treatment_no!=. 
replace dec = 9 if week_awd_adj2>=p80 & week_awd_adj2< p90 & week_awd_adj2!=.  & treatment_no!=. 
replace dec =10 if week_awd_adj2>=p90                      & week_awd_adj2!=.  & treatment_no!=. 

* define omitted categories (by dropping dec10)
* this will also establish the relevant omitted categories for the interactions below
tab dec, gen(dec_)
drop dec_10


* Create an hours worked variable that excludes outliers

sum     dln_hrs if treatment_no>=1 & treatment_no<=12 & awd==1 & dec!=. & d==1, detail
gen     dln_hrsz = (dln_hrs-r(mean))/r(sd) if treatment_no>=1 & treatment_no<=12 & dec!=. & d==1
sum     dln_hrs if treatment_no>=1 & treatment_no<=12 & awd==1 & dec!=. & d==0, detail
replace dln_hrsz = (dln_hrs-r(mean))/r(sd) if treatment_no>=1 & treatment_no<=12 & dec!=. & d==0

gen     outlier = 0 if dln_hrsz!=.
replace outlier = 1 if abs(dln_hrsz)>4 & dln_hrsz!=.

bysort j_id treatment_no: egen outlier_max = max(outlier)

gen     dln_hrs_out = dln_hrs
replace dln_hrs_out = .      if outlier_max==1


/**************************************************************
* Descriptives
**************************************************************/

* Construct percentiles based on 'matched' jobs. That is, jobs that are active at the start and end of each 6m transition

drop p10 p20 p30 p40 p50 p60 p70 p80 p90

forvalues s=10(10)90 {
bysort treatment_no d: egen p`s' = pctile(week_awd_adj2) if awd==1 & treatment_no!=. & dln_wage!=. & dln_hrs!=., p(`s')
}
sort  j_id surveyq

gen dec_m=.
replace dec_m = 1 if week_awd_adj2< p10                      & week_awd_adj2!=.  & treatment_no!=. & dln_wage!=. & dln_hrs!=.
replace dec_m = 2 if week_awd_adj2>=p10 & week_awd_adj2< p20 & week_awd_adj2!=.  & treatment_no!=. & dln_wage!=. & dln_hrs!=.
replace dec_m = 3 if week_awd_adj2>=p20 & week_awd_adj2< p30 & week_awd_adj2!=.  & treatment_no!=. & dln_wage!=. & dln_hrs!=.
replace dec_m = 4 if week_awd_adj2>=p30 & week_awd_adj2< p40 & week_awd_adj2!=.  & treatment_no!=. & dln_wage!=. & dln_hrs!=.
replace dec_m = 5 if week_awd_adj2>=p40 & week_awd_adj2< p50 & week_awd_adj2!=.  & treatment_no!=. & dln_wage!=. & dln_hrs!=.
replace dec_m = 6 if week_awd_adj2>=p50 & week_awd_adj2< p60 & week_awd_adj2!=.  & treatment_no!=. & dln_wage!=. & dln_hrs!=.
replace dec_m = 7 if week_awd_adj2>=p60 & week_awd_adj2< p70 & week_awd_adj2!=.  & treatment_no!=. & dln_wage!=. & dln_hrs!=.
replace dec_m = 8 if week_awd_adj2>=p70 & week_awd_adj2< p80 & week_awd_adj2!=.  & treatment_no!=. & dln_wage!=. & dln_hrs!=.
replace dec_m = 9 if week_awd_adj2>=p80 & week_awd_adj2< p90 & week_awd_adj2!=.  & treatment_no!=. & dln_wage!=. & dln_hrs!=.
replace dec_m =10 if week_awd_adj2>=p90                      & week_awd_adj2!=.  & treatment_no!=. & dln_wage!=. & dln_hrs!=.

gen full = 1 if treatment_no>=1 & treatment_no<=11

bysort treatment_no: tabstat death2 if treatment_no!=. & awd==1 & full==1 & dec!=. & d==0, by(dec) statistics(mean sd N) 
bysort treatment_no: tabstat death2 if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, by(dec) statistics(mean sd N) 

bysort treatment_no: tabstat dln_wage if treatment_no!=. & awd==1 & full==1 & dec!=. & d==0, by(dec_m) statistics(mean sd N) 
bysort treatment_no: tabstat dln_wage if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, by(dec_m) statistics(mean sd N) 

bysort treatment_no: tabstat dln_hrs if treatment_no!=. & awd==1 & full==1 & dec!=. & d==0, by(dec_m) statistics(mean sd N) 
bysort treatment_no: tabstat dln_hrs if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, by(dec_m) statistics(mean sd N) 

bysort treatment_no: tabstat dln_hrs_out if treatment_no!=. & awd==1 & full==1 & dec!=. & d==0, by(dec_m) statistics(mean sd N) 
bysort treatment_no: tabstat dln_hrs_out if treatment_no!=. & awd==1 & full==1 & dec!=. & d==1, by(dec_m) statistics(mean sd N) 

* For calculating the average job destruction rate by wage decile:
tabstat death2 if treatment_no!=. & awd==1 & full==1 & dec!=. & (d==0 | d==1), by(dec) statistics(mean sd N) 

drop full p10 p20 p30 p40 p50 p60 p70 p80 p90 dec_m


/**************************************************************
* Construct interaction terms for DD and DDD models
**************************************************************/

* treatment period dummy X wage decile 
forvalues c = 1(1)9 {
gen dXdec_`c'=d*dec_`c'
}
* treatment_no X wage decile (allows for time-varying group-level heterogeneity)
tab treatment_no, gen(tr_)
forvalues p = 1(1)18  {
forvalues c = 1(1)9 {
gen tr_`p'Xdec_`c'=tr_`p'*dec_`c'
}
}
* awd X quarter  
forvalues c = 1(1)35 {
gen awdXqtr_`c'=awd*qtr_`c'
}
* awd X wage decile  
forvalues c = 1(1)9 {
gen awdXdec_`c'=awd*dec_`c'
}
* awd X treatment_no X wage decile 
forvalues p = 1(1)18  {
forvalues c = 1(1)9 {
gen awdXtr_`p'Xdec_`c'=tr_`p'*dec_`c'*awd
}
drop tr_`p'
}
* awd X treatment period dummy X wage decile (DDD coefficient in pooled model)
forvalues c = 1(1)9 {
gen awdXdXdec_`c'=awd*d*dec_`c'
}
*

/**************************************************************
 Variables in continuous treatment specification
**************************************************************/

* Continuous treatment specification (DD)
gen c_treat = snr_adj2

gen dXc_treat = d * c_treat

* Treatment_no X c_treat (allows for time-varying group-level heterogeneity)
tab treatment_no, gen(tr_)
forvalues p = 1(1)18  {
gen c_treatXtr_`p't=tr_`p'*c_treat
drop tr_`p'
}
*
* Continuous treatment specification (DDD)
gen awdXc_treat = awd * c_treat
gen awdXdXc_treat = awd * d * c_treat

* Treatment_no X c_treat (allows for time-varying group-level heterogeneity in DDD)
tab treatment_no, gen(tr_)
forvalues p = 1(1)18  {
gen c_treatXawdXtr_`p't=tr_`p'*c_treat*awd
drop tr_`p'
}
*

save private_transitions, replace

/**************************************************************
* Categorical treatment specification (DD and DDD)
**************************************************************/

cd "\\sasnasprd\SASData\W386\RBA\dta_files\master"

use private_transitions, clear

cd "\\sasnasprd\SASData\W386\RBA\analysis\STATA_output"

local clust j_id

gen full = 1 if treatment_no>=1 & treatment_no<=11
gen plac = 1 if treatment_no==12
gen pmek = 1 if treatment_no>=5 & treatment_no<=11

set matsize 1000

foreach s in full plac pmek {

matrix A`s' = J(19,15,.)
matrix colnames A`s' = dd_w_a dd_w_e ddd_w dd_h_a dd_h_e ddd_h dd_s1_a dd_s1_e ddd_s1 dd_s2_a dd_s2_e ddd_s2 dd_ho_a dd_ho_e ddd_ho
matrix rownames A`s' = b_1 b_2 b_3 b_4 b_5 b_6 b_7 b_8 b_9 se_1 se_2 se_3 se_4 se_5 se_6 se_7 se_8 se_9 obs

matrix B`s' = J(3,15,.)
matrix colnames B`s' = dd_w_a dd_w_e ddd_w dd_h_a dd_h_e ddd_h dd_s1_a dd_s1_e ddd_s1 dd_s2_a dd_s2_e ddd_s2 dd_ho_a dd_ho_e ddd_ho
matrix rownames B`s' = b_ se_ obs

local i = 0

foreach x in dln_wage dln_hrs death1 death2 dln_hrs_out {

* Percentile treatments

quietly reg `x' qtr_* dec_* dXdec_*                                      if treatment_no!=. & awd==1 & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,1+`i'] =  _b[dXdec_`z']*100
matrix A`s'[`z'+9,1+`i'] =  _se[dXdec_`z']*100
matrix A`s'[19,   1+`i'] = e(N)
}
quietly reg `x' qtr_* dec_* dXdec_*                                      if treatment_no!=. & awd==0 & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,2+`i'] =  _b[dXdec_`z']*100
matrix A`s'[`z'+9,2+`i'] =  _se[dXdec_`z']*100
matrix A`s'[19,   2+`i'] = e(N)
}
quietly reg `x' qtr_* dec_* dXdec_* awd awdXqtr_* awdXdec_* awdXdXdec_* if treatment_no!=. & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,3+`i'] =  _b[awdXdXdec_`z']*100
matrix A`s'[`z'+9,3+`i'] =  _se[awdXdXdec_`z']*100
matrix A`s'[19,   3+`i'] = e(N)
}
*

* Continuous treatment

quietly reg `x' qtr_* c_treat dXc_treat                                         if treatment_no!=. & awd==1 & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,1+`i'] =  _b[dXc_treat]
matrix B`s'[2,1+`i'] =  _se[dXc_treat]
matrix B`s'[3,1+`i'] = e(N)

quietly reg `x' qtr_* c_treat dXc_treat                                         if treatment_no!=. & awd==0 & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,2+`i'] =  _b[dXc_treat]
matrix B`s'[2,2+`i'] =  _se[dXc_treat]
matrix B`s'[3,2+`i'] = e(N)

quietly reg `x' qtr_* c_treat dXc_treat awd awdXqtr_* awdXc_treat awdXdXc_treat if treatment_no!=.          & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,3+`i'] =  _b[awdXdXc_treat]
matrix B`s'[2,3+`i'] =  _se[awdXdXc_treat]
matrix B`s'[3,3+`i'] = e(N)

local i = `i'+3
}
}
*

putexcel A1=matrix(Afull, names) using results_transitions.xlsx, sheet("decile_full") replace 
putexcel A1=matrix(Bfull, names) using results_transitions.xlsx, sheet("contin_full") modify 
putexcel A1=matrix(Aplac, names) using results_transitions.xlsx, sheet("decile_placebo") modify 
putexcel A1=matrix(Bplac, names) using results_transitions.xlsx, sheet("contin_placebo") modify 
putexcel A1=matrix(Apmek, names) using results_transitions.xlsx, sheet("decile_pmek") modify 
putexcel A1=matrix(Bpmek, names) using results_transitions.xlsx, sheet("contin_pmek") modify 


/************************************************************************************
 Estimates by year
*************************************************************************************/

foreach i in dd ddd {
matrix T_`i' = J(12,15,.)
matrix colnames T_`i' = w_b w_se w_N h_b h_se h_N s1_b s1_se s1_N s2_b s2_se s2_N ho_b ho_se ho_N
matrix rownames T_`i' = 1998 1999 2000 2001 2002 2003 2004 2005 2006 2007 2008 2009
}
*

local i = 0

foreach z in dln_wage dln_hrs death1 death2 dln_hrs_out {

* DD estimates by year (seperate regressions)
forvalues x = 1(1)12 {
quietly reg `z' d c_treat dXc_treat if treatment_no==`x' & awd==1 & dec!=., cluster(`clust') noomitted
matrix T_dd[`x',1+`i'] =  _b[dXc_treat]
matrix T_dd[`x',2+`i'] =  _se[dXc_treat]
matrix T_dd[`x',3+`i'] = e(N)
}

* DDD estimates by year (seperate regressions)
forvalues x = 1(1)12 {
quietly reg `z' d c_treat dXc_treat awd awd#d awdXc_treat awdXdXc_treat if treatment_no==`x' & dec!=., cluster(`clust') noomitted
matrix T_ddd[`x',1+`i'] =  _b[awdXdXc_treat]
matrix T_ddd[`x',2+`i'] =  _se[awdXdXc_treat]
matrix T_ddd[`x',3+`i'] = e(N)

}
local i = `i'+3
}
*

putexcel A1=matrix(T_dd , names) using results_transitions.xlsx, sheet("timevarying_dd")  modify 
putexcel A1=matrix(T_ddd, names) using results_transitions.xlsx, sheet("timevarying_ddd") modify 


* Hypothesis tests (ignore the placebo period). Test that elasticities are equal over time in DDD model.

* Treatment_no X c_treat X d 
tab treatment_no, gen(tr_)
forvalues p = 1(1)18  {
gen dXc_treatXtr_`p'=dXc_treat*tr_`p'
gen awdXc_treatXtr_`p'=awdXc_treat*tr_`p'
gen awdXdXc_treatXtr_`p' = awdXdXc_treat*tr_`p' 
drop tr_`p'
}
*

* DDD
foreach z in dln_wage dln_hrs death1 death2 dln_hrs_out {
quietly reg `z' qtr_* c_treat c_treatXtr_* dXc_treat dXc_treatXtr_* awd awdXqtr_* awdXc_treat awdXc_treatXtr_* awdXdXc_treat awdXdXc_treatXtr_3-awdXdXc_treatXtr_11 if treatment_no!=.  & full==1 & dec!=. , cluster(`clust') noomitted
test (awdXdXc_treatXtr_3 awdXdXc_treatXtr_4 awdXdXc_treatXtr_5 awdXdXc_treatXtr_6 awdXdXc_treatXtr_7 awdXdXc_treatXtr_8 awdXdXc_treatXtr_9 awdXdXc_treatXtr_10 awdXdXc_treatXtr_11)
}
*

drop dXc_treatXtr_* awdXc_treatXtr_* awdXdXc_treatXtr_*

/************************************************************************************
 Robustness tests
 
 1- Allow for time-varying group-level heterogeneity in decile treatment specification
 
    [add tr_* and awdXtr_* to the percentile treatment specification]

	[add c_treatXtr_* and c_treatXawdXtr_* to the continuous treatment specification]

 2- Is continuous specification robust to controlling for decile dummies
 
 3- Excluding outliers (in terms of log changes)
 
 4- Introduce controls
*************************************************************************************/

* 1 - Allow for time-varying group-level heterogeneity

foreach s in full plac pmek {

matrix A`s' = J(19,12,.)
matrix colnames A`s' = dd_w_a dd_w_e ddd_w dd_h_a dd_h_e ddd_h dd_s1_a dd_s1_e ddd_s1 dd_s2_a dd_s2_e ddd_s2
matrix rownames A`s' = b_1 b_2 b_3 b_4 b_5 b_6 b_7 b_8 b_9 se_1 se_2 se_3 se_4 se_5 se_6 se_7 se_8 se_9 obs

matrix B`s' = J(3,12,.)
matrix colnames B`s' = dd_w_a dd_w_e ddd_w dd_h_a dd_h_e ddd_h dd_s1_a dd_s1_e ddd_s1 dd_s2_a dd_s2_e ddd_s2 
matrix rownames B`s' = b_ se_ obs

local i = 0

foreach x in dln_wage dln_hrs death1 death2 {

* Percentile treatments

quietly reg `x' qtr_* dec_* tr_* dXdec_*                                      if treatment_no!=. & awd==1 & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,1+`i'] =  _b[dXdec_`z']*100
matrix A`s'[`z'+9,1+`i'] =  _se[dXdec_`z']*100
matrix A`s'[19,   1+`i'] = e(N)
}
quietly reg `x' qtr_* dec_* tr_* dXdec_*                                      if treatment_no!=. & awd==0 & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,2+`i'] =  _b[dXdec_`z']*100
matrix A`s'[`z'+9,2+`i'] =  _se[dXdec_`z']*100
matrix A`s'[19,   2+`i'] = e(N)
}
quietly reg `x' qtr_* dec_* tr_* dXdec_* awd awdXqtr_* awdXdec_* awdXtr_* awdXdXdec_* if treatment_no!=. & `s'==1 & dec!=., cluster(`clust') noomitted
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,3+`i'] =  _b[awdXdXdec_`z']*100
matrix A`s'[`z'+9,3+`i'] =  _se[awdXdXdec_`z']*100
matrix A`s'[19,   3+`i'] = e(N)
}
*

* Continuous treatment

quietly reg `x' qtr_* c_treat c_treatXtr_* dXc_treat                                         if treatment_no!=. & awd==1 & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,1+`i'] =  _b[dXc_treat]
matrix B`s'[2,1+`i'] =  _se[dXc_treat]
matrix B`s'[3,1+`i'] = e(N)

quietly reg `x' qtr_* c_treat c_treatXtr_* dXc_treat                                         if treatment_no!=. & awd==0 & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,2+`i'] =  _b[dXc_treat]
matrix B`s'[2,2+`i'] =  _se[dXc_treat]
matrix B`s'[3,2+`i'] = e(N)

quietly reg `x' qtr_* c_treat c_treatXtr_* dXc_treat c_treatXawdXtr_* awd awdXqtr_* awdXc_treat awdXdXc_treat if treatment_no!=. & `s'==1 & dec!=. , cluster(`clust') noomitted
matrix B`s'[1,3+`i'] =  _b[awdXdXc_treat]
matrix B`s'[2,3+`i'] =  _se[awdXdXc_treat]
matrix B`s'[3,3+`i'] = e(N)

local i = `i'+3
}
}
*
putexcel A1=matrix(Afull, names) using results_transitions.xlsx, sheet("decile_full_tvuh") modify 
putexcel A1=matrix(Bfull, names) using results_transitions.xlsx, sheet("contin_full_tvuh") modify 
putexcel A1=matrix(Aplac, names) using results_transitions.xlsx, sheet("decile_placebo_tvuh") modify 
putexcel A1=matrix(Bplac, names) using results_transitions.xlsx, sheet("contin_placebo_tvuh") modify 
putexcel A1=matrix(Apmek, names) using results_transitions.xlsx, sheet("decile_pmek_tvuh") modify 
putexcel A1=matrix(Bpmek, names) using results_transitions.xlsx, sheet("contin_pmek_tvuh") modify 


/******************************************************************************************
 Control for firm ID FE and its interaction with d
 
 Equivalent to grouping on firm ID and D
 
 In transitions specification, cannot estimate this using individual FD because no longer working with a balanced panel
******************************************************************************************/

encode f_unitid, gen(firm)

egen firm_by_d = group(d firm)

local clust j_id

foreach s in full pmek plac {

matrix A`s' = J(19,5,.)
matrix colnames A`s' = dd_w_a  dd_h_a dd_s1_a dd_s2_a dd_ho_a
matrix rownames A`s' = b_1 b_2 b_3 b_4 b_5 b_6 b_7 b_8 b_9 se_1 se_2 se_3 se_4 se_5 se_6 se_7 se_8 se_9 obs

matrix B`s' = J(3,5,.)
matrix colnames B`s' = dd_w_a  dd_h_a dd_s1_a dd_s2_a dd_ho_a
matrix rownames B`s' = b_ se_ obs

local i = 0

foreach x in dln_wage dln_hrs death1 death2 dln_hrs_out {

* Percentile treatments (log-change specification)

quietly areg `x' qtr_* dec_* dXdec_*   if treatment_no!=. & `s'==1  & awd==1 & dec!=., cluster(`clust') noomitted absorb(firm_by_d)
forvalues z = 1(1)9 {
matrix A`s'[`z'  ,1+`i'] =  _b[dXdec_`z']*100
matrix A`s'[`z'+9,1+`i'] =  _se[dXdec_`z']*100
matrix A`s'[19,   1+`i'] = e(N)
}

* Continuous treatment (log-change specification)

quietly areg `x' qtr_* c_treat dXc_treat if treatment_no!=. & `s'==1 & awd==1 & dec!=., cluster(`clust') noomitted absorb(firm_by_d)
matrix B`s'[1,1+`i'] =  _b[dXc_treat]
matrix B`s'[2,1+`i'] =  _se[dXc_treat]
matrix B`s'[3,1+`i'] = e(N)

local i = `i'+1
}
}
*
putexcel A1=matrix(Afull, names) using results_transitions.xlsx, sheet("decile_full_controls") modify 
putexcel A1=matrix(Bfull, names) using results_transitions.xlsx, sheet("contin_full_controls") modify 
putexcel A1=matrix(Apmek, names) using results_transitions.xlsx, sheet("decile_pmek_controls") modify 
putexcel A1=matrix(Bpmek, names) using results_transitions.xlsx, sheet("contin_pmek_controls") modify 
putexcel A1=matrix(Aplac, names) using results_transitions.xlsx, sheet("decile_placebo_controls") modify 
putexcel A1=matrix(Bplac, names) using results_transitions.xlsx, sheet("contin_placebo_controls") modify 


/******************************************************************************************
 Robustness checks and additional results
******************************************************************************************/

* Exclude Queensland (due to timing issues)

foreach x in death2 {
reg `x' qtr_* c_treat      dXc_treat  if treatment_no!=. & awd==1 & full==1 & dec!=. & f_stateest!=3, cluster(j_id) noomitted
reg `x' qtr_* c_treat      dXc_treat  if treatment_no!=. & awd==0 & full==1 & dec!=. & f_stateest!=3, cluster(j_id) noomitted
reg `x' qtr_* c_treat dXc_treat awd awdXqtr_* awdXc_treat awdXdXc_treat if treatment_no!=.  & full==1 & dec!=. & f_stateest!=3, cluster(j_id) noomitted
}
*

* end of do file
