/**************************************************************/
/*  Program created by James Bishop during RBA inposting in  **/
/*  April 2016. It creates a STATA database of the WPI micro **/
/*  data.                                                    **/

/*  NB: Run extract_data.sas first                           **/
/**************************************************************/

clear all

set more off

cd "\\sasnasprd\SASData\W386\RBA\dta_files\mujaa90"


/**************************************************************/
* Append the four separate files to create master job-level file
use mujaa90_Mar_temp, clear
append using mujaa90_Jun_temp
append using mujaa90_Sep_temp
append using mujaa90_Dec_temp

sort unitid surveyq

save mujaa90_master, replace

*erase jobdata_Mar_temp.dta
*erase jobdata_Jun_temp.dta
*erase jobdata_Sep_temp.dta
*erase jobdata_Dec_temp.dta

/**************************************************************/

egen  id = group(unitid)
sort  id surveyq
xtset id surveyq

/**************************************************************
 Underlying calculations for derived variables in mujaa90:
 
 - 'basepay' (base rate of pay, per pay period)
 
 gen     basepay = 0 if reordge==0
 replace basepay = reordge - shfalw - totoalw if basepay==.
 replace basepay = 0                          if basepay <0
 
 
 - 'hbsep' (hourly base rate of pay)
 
  gen hbsep = basepay / ordhrsd
  
 - 'pqhbsep' (percentage change in hourly base rate of pay relative to previous quarter)
 
  gen     pqhbsep = (hbsep/L1.hbsep)*100-100 if hbsep!=0
  replace pqhbsep = 999 if L1.hbsep==0

 - 'ordhrswk' (ordinary hours worked, weekly equivalent) 
  
  gen ordhrswk = ordhrsd / 1        if ordgefq=="W"
  gen ordhrswk = ordhrsd / 2        if ordgefq=="F"
  gen ordhrswk = ordhrsd / 4.333333 if ordgefq=="M"
  gen ordhrswk = stdhrsd            if ordgefq=="O"
  
  NB: WPI team set ordhrswk equal to weekly standard hours when 'other' pay frequency. Better to 
  use the text string 'ordgefqo' to do something more precise. Or just assume that a ratio ordhrsd/stdhrsd
  continues to hold over time.
  
 **************************************************************/

 gen implicit_divisor = ordhrsd/ordhrswk 
 
 sum implicit_divisor if ordgefq=="W" | payfreq=="W"
 sum implicit_divisor if ordgefq=="F" | payfreq=="F" 
 sum implicit_divisor if ordgefq=="M" | payfreq=="M" 
 sum implicit_divisor if ordgefq=="O" | payfreq=="O" 
 
 gen     pay_divisor = 1        if ordgefq=="W" | payfreq=="W"
 replace pay_divisor = 2        if ordgefq=="F" | payfreq=="F"
 replace pay_divisor = 4.333333 if ordgefq=="M" | payfreq=="M"
  
 sort unitid surveyq
 
 by   unitid:  egen stdhrsd_mn = mean(stdhrsd) if ordgefq=="O" | payfreq=="O"
 by   unitid:  egen ordhrsd_mn = mean(ordhrsd) if ordgefq=="O" | payfreq=="O"

 replace pay_divisor = ordhrsd_mn/stdhrsd_mn if ordgefq=="O" | payfreq=="O"

 gen ordhrswk_jb = ordhrsd/pay_divisor
 
 
* Generate base rate ex. estimated casual loading
gen     hbsep_excs = hbsep
replace hbsep_excs = hbsep/(1+0.2) if ftptflg=="C" | casload=="Y"
  
keep  merge_id surveyq basepay hbsep hbsep_excs ordhrsd stdhrsd ordhrswk ordhrswk_jb pay_divisor casload ftptflg pmekflg pti signiflg imprc splice splicerc redun1 redun2 changeei temp_dead defnilqt dfnilcmt vpayflg
order merge_id surveyq basepay hbsep hbsep_excs ordhrsd stdhrsd ordhrswk ordhrswk_jb pay_divisor casload ftptflg pmekflg pti signiflg imprc splice splicerc redun1 redun2 changeei temp_dead defnilqt dfnilcmt vpayflg
 
* Give all mujaa90 job-level variables an m_ prefix
foreach var of varlist _all {
rename `var' m_`var'
}
* 
rename m_merge_id merge_id
rename m_surveyq  surveyq
sort   merge_id   surveyq

save mujaa90_master, replace

* end of do file

