/*******************************************************************************

	6_AAWI.do
	
	This file contains code to replicate the Department of Jobs and Small
	Business' measure of average annualised wage increases (AAWI) for analysis
	in Bishop J and I Chan (2019), Is Declining Union Membership Contributing
	to Low Wages Growth?, RBA Research Discussion Paper No 2019-02.
	It also examines renegotiation lags for Section 4.4 of the paper.
	
*******************************************************************************/

clear all
set more off   

local dir  "<path for 'data' folder here>"

cd "`dir'"

use wad_extract_clean_panel_firm_setup, clear

/*********************************************************************************************
* Renegotiation lags
 *******************************************************************************************/ 

sort id fam_count

gen end_date_l1 = L1.end_date 

format end_date_l1 %td

gen reneg_lag = (start_date - end_date_l1) / 365

replace reneg_lag=0 if reneg_lag<0

* Topcoded renegotiation lag at 1 year and 18 months

gen     reneg_lagtc12=reneg_lag
replace reneg_lagtc12=1    if reneg_lag>1 & reneg_lag!=.

gen     reneg_lagtc18=reneg_lag
replace reneg_lagtc18=1.5   if reneg_lag>1.5 & reneg_lag!=.

* There is one EBA (AE882924) that covers a large numnber of employees and has a substantial renegotiation lag.
* If this is not excluded, there is a large spike in the weighted mean renegotiation lag in 2010q4.
* Note that these negotiation lags are measured at the time the new agreement starts, rather than the delay on the current stock

tabstat reneg_lag if public==0 & eba_id!="AE882924", by(surveyq) statistics(mean count) 
tabstat reneg_lag if public==1 & eba_id!="AE882924", by(surveyq) statistics(mean count) 

tabstat reneg_lag                             if eba_id!="AE882924", by(surveyq) statistics(mean p10 p25 p50 p75 p90 count) 
tabstat reneg_lag [weight=employees_combined] if eba_id!="AE882924", by(surveyq) statistics(mean p10 p25 p50 p75 p90 count) 

/*********************************************************************************************
* Agreement duration

  To replicate the DoE's measure of AAWI, we need to replicate their measure of duration.

  Calculate 'effective duration' as per Dept of Employment
  
  Consider two measures, one that sets effective duration to 1yr if formal duration <1yr, and one that does not.  
 
  Over the period from 2011 onwards, some agreements have a shorter duration because the first wage increase under 
  the agreements concerned cannot be quantified (ie turned into a % amount) so the DoE have calculated an AAWI for
  the rest of the agreement and used the date of the second wage increase under the agt as the start date for
  calculating the duration of the agreement.
  NB: for such agreements, the date of the second increase is actually coded as firs_incr rather than secd_incr
*******************************************************************************************/ 
 
replace duration = . if openduration=="Yes"

gen     firs_nc = 0
replace firs_nc = 1 if (firstincnotquant_otherreason=="Yes" | firstincnotquant_classsiftruct=="Yes") 

* Formal duration
 
gen     dur_formal = (expi_date - cert_date)/365

/* Effective duration
   As per Dept of Employment
   DoE documentation suggests that set effective duration to 1yr if formal duration <1yr. However, the replication comes
   closer if we set effective duration to 1yr if effective duration <1yr 
   
   To replicate DoE:
   
   gen     dur_effect = (min(term_date, max(expi_date, last_incr))-min(cert_date, comm_date, firs_incr))/365  if firs_nc==0
   replace dur_effect = (min(term_date, max(expi_date, last_incr))-firs_incr)/365                             if firs_nc==1
   replace dur_effect =1 if dur_effect<1
   
*/
   
* modified approach that only uses information available at the time the agreement is signed:

gen     dur_effect = (max(expi_date, last_incr)-min(cert_date, comm_date, firs_incr))/365  if firs_nc==0
replace dur_effect = (max(expi_date, last_incr)-firs_incr)/365                             if firs_nc==1
replace dur_effect = 1 if dur_effect<1

* To replicate DoE
   
gen     dur_effect_doe = (min(term_date, max(expi_date, last_incr))-min(cert_date, comm_date, firs_incr))/365  if firs_nc==0
replace dur_effect_doe = (min(term_date, max(expi_date, last_incr))-firs_incr)/365                             if firs_nc==1
replace dur_effect_doe = 1 if dur_effect_doe<1

/* Duration measure accounting for renegotiation delay (topcoded at 12 months)
   AAWI based on this measure will be missing for the first EBA in the family */

gen dur_reneg = dur_effect + reneg_lagtc12

/*===========================================================================
  Calculate AAWI using Dept of Employment Formula
  
     Construct one set of AAWI measures that include conditional increases and another that excludes them
  
        aawi_a  denotes aawi including conditional increases
  
        aawi_b  denotes aawi excluding conditional increases
  
     In cases where the wage increase is available in dollar terms the DOE has calculated a AAWI
	 (likely using data on the wage levels in the agreement). For these, just use the DOE's measure of AAWI.
	 
	 Our measure (using the modified approach for dur_effect aobve) does not adjust for term_date as this is
	 not known at the time of the contract
===========================================================================*/

forvalues z = 1(1)18 {
gen ia`z' = 1+incr_pc`z'/100

gen ib`z'    = ia`z'
replace ib`z'= 1 if conditional`z'=="Yes"

encode conditional`z', gen(cc`z')
}
*
gen  incr_tota = incr_tot
egen incr_totb = anycount(cc*), values(1) 

foreach y in effect reneg {

foreach x in a b {

gen aawi_`x'_`y' = .
replace aawi_`x'_`y' = ((i`x'1)^(1/dur_`y')-1)*100 if incr_tot`x'==1 
replace aawi_`x'_`y' = ((i`x'1*i`x'2)^(1/dur_`y')-1)*100 if incr_tot`x'==2 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3)^(1/dur_`y')-1)*100 if incr_tot`x'==3 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4)^(1/dur_`y')-1)*100 if incr_tot`x'==4 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5)^(1/dur_`y')-1)*100 if incr_tot`x'==5 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6)^(1/dur_`y')-1)*100 if incr_tot`x'==6 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7)^(1/dur_`y')-1)*100 if incr_tot`x'==7 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8)^(1/dur_`y')-1)*100 if incr_tot`x'==8 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8*i`x'9)^(1/dur_`y')-1)*100 if incr_tot`x'==9 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8*i`x'9*i`x'10)^(1/dur_`y')-1)*100 if incr_tot`x'==10 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8*i`x'9*i`x'10*i`x'11)^(1/dur_`y')-1)*100 if incr_tot`x'==11 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8*i`x'9*i`x'10*i`x'11*i`x'12)^(1/dur_`y')-1)*100 if incr_tot`x'==12 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8*i`x'9*i`x'10*i`x'11*i`x'12*i`x'13)^(1/dur_`y')-1)*100 if incr_tot`x'==13 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8*i`x'9*i`x'10*i`x'11*i`x'12*i`x'13*i`x'14)^(1/dur_`y')-1)*100 if incr_tot`x'==14 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8*i`x'9*i`x'10*i`x'11*i`x'12*i`x'13*i`x'14*i`x'15)^(1/dur_`y')-1)*100 if incr_tot`x'==15 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8*i`x'9*i`x'10*i`x'11*i`x'12*i`x'13*i`x'14*i`x'15*i`x'16)^(1/dur_`y')-1)*100 if incr_tot`x'==16 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8*i`x'9*i`x'10*i`x'11*i`x'12*i`x'13*i`x'14*i`x'15*i`x'16*i`x'17)^(1/dur_`y')-1)*100 if incr_tot`x'==17 
replace aawi_`x'_`y' = ((i`x'1*i`x'2*i`x'3*i`x'4*i`x'5*i`x'6*i`x'7*i`x'8*i`x'9*i`x'10*i`x'11*i`x'12*i`x'13*i`x'14*i`x'15*i`x'16*i`x'17*i`x'18)^(1/dur_`y')-1)*100 if incr_tot`x'==18 
}
}
* 

/* For agreements with flat dollar increases, use the DOE's calculations of AAWI
   
   DoE: (Flat dollar increases are converted to a percentage using average weekly ordinary time earnings (AWOTE),
    drawn from ABS, 6302.0 - Average Weekly Earnings, Australia) for the relevant ANZSIC industry division and quarter)
	
*/ 

egen    incr_amt_tot = rowtotal(incr_amount*)  

gen     flatdollar = 0
replace flatdollar = 1 if incr_amt_tot>0 & incr_amt_tot!=.

* Back out the DoE's calculation of the total cumulative increase i.e. the sum of the (1+r) terms

gen aawi_cuminc = [(aawi/100) + 1]^dur_effect_doe

* Apply my measures of duration to recompute AAWI

replace aawi_a_effect = ((aawi_cuminc)^(1/dur_effect)-1)*100 if flatdollar==1
replace aawi_b_effect = ((aawi_cuminc)^(1/dur_effect)-1)*100 if flatdollar==1
replace aawi_a_reneg  = ((aawi_cuminc)^(1/dur_reneg )-1)*100 if flatdollar==1
replace aawi_b_reneg  = ((aawi_cuminc)^(1/dur_reneg )-1)*100 if flatdollar==1

* Examine how closely we can replicate te DoE's measure - NB: we use a different measure of duration, see above

               tabstat aawi aawi_a_effect aawi_b_effect [weight=employees_combined] if aawi!=0 & aawi_a_effect!=., by(cert_date_qy)
			   
bysort public: tabstat aawi aawi_a_effect aawi_b_effect [weight=employees_combined] if aawi!=0 & aawi_a_effect!=., by(cert_date_qy)

* Examine whether accounting for renegotiation lags makes a difference

tabstat aawi_a_effect aawi_a_reneg if aawi_a_effect!=. & aawi_a_reneg!=. [weight=employees_combined], by(surveyq) 

save wad_extract_final_working, replace

sort eba_id cert_date

drop incr_pc* incr_amount* non_compound* ia* ib* cc* conditional* incr_date_qy* incr_date* incr_tota incr_totb incr_amt_tot flatdollar 

save wad_extract_final, replace

/*********************************************************************************************
  * Consistent series of 2-digit industry
  
    Use dual-coded Census to map ANZSIC93 codes to ANZSIC06 codes
 *******************************************************************************************/ 

import delimited 2006_census_anzsic93_anzsic06.csv, rowrange(10:7749) colrange(2:4) varnames(10) clear

gen byte notnumeric = real(industryofemploymentanzsic93ind9)==. | real(industryofemploymentanzsic06ind0)==.

drop if notnumeric==1
drop    notnumeric

rename industryofemploymentanzsic93ind9 anzsic93
rename industryofemploymentanzsic06ind0 anzsic06

destring anzsic93 anzsic06, replace

bysort anzsic93: egen count_sum = sum(count)
gen                   count_sh  = count / count_sum * 100

* Identify most relevant anzsic06 for each anzsic93 

bysort anzsic93: egen count_max = max(count)

keep if count==count_max

keep anzsic93 anzsic06

rename anzsic06 anzsic06_common

save anzsic_mapping, replace

use wad_extract_final, clear

merge m:1 anzsic93 using anzsic_mapping
drop if _merge==2
drop    _merge

gen     anzsic06_2dig = anzsic06
replace anzsic06_2dig = anzsic06_common if anzsic06_2dig==.

erase wad_extract_final_working.dta

save wad_extract_final, replace

clear 

* end of do file
