version 13.0 
clear all
capture log close

local savepath "G:\Research\TrentS\Leaning against the wind\Code_publication\Stata_crisis_probability"
cd "`savepath'"
log using creditgap.smcl, replace

********************************************************************************
*Progam name: creditgap.do
*Log name: creditgap_data.smcl
*Created by Matt Read
*Purpose: Using dataset from Schularick and Taylor (2012), calculate the 
*detrended credit-to-GDP ratio. 
********************************************************************************

set more off, perm
set matsize 8000

*Loop through countries to HP filter credit-to-GDP ratio because state-space
*function can't be run on a panel
local countries ""AUS" "CAN" "CHE" "DEU" "DNK" "ESP" "FRA" "GBR" "ITA" "JPN" "NLD" "NOR" "SWE" "USA""
use panel17.dta, clear

foreach x of local countries {

	*Load dataset
	use panel17.dta, clear
	keep if iso == "`x'"

	sort year iso
	tsset year
	
	* omit 2009
	drop if year==2009

	if "`x'" == "AUS" {
		*Import updated data for Australia
		merge 1:1 year iso using updated_aus_data.dta, nogen
		sort year iso
		replace loans1 = loans_aus if iso == "AUS"
		replace cpi = cpi_aus if iso == "AUS"
		replace gdp = gdp_aus if iso == "AUS"
		replace crisisST = 0 if iso == "AUS" & year > 2008
	}

	*Do not use noisy data on French loans before 1895 (consistent with
	*Schularick and Taylor (2012))
	if "`x'" == "FRA" {
	
		replace loans1 = . if year<=1895
	
	}

	* make some series
	gen lloans = log(loans1) 
	gen dlloans = d.lloans
	gen loansgdp = loans1/gdp*100

	*Linearly interpolate missing data on credit-to-GDP ratio
	ipolate loansgdp year, generate(credgdp)

	*Apply one-sided HP filter to credit-to-GDP ratio
	*Need to use sspace command to estimate HP filter in state-space form, 
	*because the inbuilt HP filter in Stata is two-sided. Filtered (or 
	*'real-time') predictions (i.e. conditional on information up to time t) 
	*from the Kalman filter will be equivalent to the one-sided filter.
	*The state-space representation is based on the fact that the HP filter can
	*be cast as a restricted local linear trend model.
	
	*Define smoothing parameter (adjusted for frequency of data)
	local lambda = 1562.5
	scalar lambda = `lambda'
	scalar sl = sqrt(1/lambda)
	*Specify constraints
	constraint drop _all
	constraint 1 [y_star]L.y_star = 1
	constraint 2 [y_star]L.beta = 1
	constraint 3 [beta]L.beta = 1
	constraint 4 [credgdp]y_star = 1
	constraint 5 [beta]e.beta = sl*[credgdp]e.credgdp
	*Estimate model
	sspace (y_star L.y_star L.beta, state noconstant) ///
	(beta L.beta e.beta, state noconstant) ///
	(credgdp y_star e.credgdp, noconstant), constraints(1/5) ///
	covstate(identity) covobserved(identity)
	*Use Kalman filter to generate estimate of 'trend' credit-to-GDP ratio. 
	*Credit-to-GDP 'gap' is the difference between the credit-to-GDP ratio and 
	*its trend.
	predict credgdp_t, states smethod(filter) equation(y_star)
	gen credgdp_gap = credgdp - credgdp_t
	gen credgdp_gap_percent = credgdp/credgdp_t*100-100
	*Select which variables to keep (subset of S&T dataset)
	keep year iso crisisST credgdp credgdp_gap lloans dlloans loansgdp ///
	gdp rgdp cpi stir
	*Save file for each country
	save `x'_credgap, replace
}

*Combine files containing individual countries' data

use AUS_credgap, clear

local countries ""CAN" "CHE" "DEU" "DNK" "ESP" "FRA" "GBR" "ITA" "JPN" "NLD" "NOR" "SWE" "USA""

foreach x of local countries {

	append using `x'_credgap.dta

}

*Erase temporary data

local countries ""AUS" "CAN" "CHE" "DEU" "DNK" "ESP" "FRA" "GBR" "ITA" "JPN" "NLD" "NOR" "SWE" "USA""

foreach x of local countries {

	erase `x'_credgap.dta
	
}

*Generate country codes and declare data as panel
egen ccode = group(iso)
tab iso, gen(ccode)
sort ccode year
xtset ccode year

*Export credit-to-GDP gap to Excel for graphing
cd "`savepath'/Graphs"
preserve

	keep year credgdp_gap iso
	*Reshape from long to wide panel
	reshape wide credgdp_gap, i(year) j(iso) string
	export excel year credgdp_gap* using "StataOutput.xlsx", ///
	sheet("credit_gap") cell("A6") sheetmodify firstrow(variables)

restore
 
drop rgdp gdp cpi stir crisisST lloans dlloans loansgdp ccode*

cd "`savepath'"
save creditgap.dta, replace

capture log close
