version 13.0 
clear all
capture log close

local savepath "G:\Research\TrentS\Leaning against the wind\Code_publication\Stata_crisis_probability"
cd "`savepath'"
log using crisis_probability.smcl, replace

********************************************************************************
*Progam name: 3_crisis_probability.do
*Log name: crisis_probability.smcl
*Written by Matt Read and Trent Saunders
*Estimate models of the probability of a financial crisis.
*The coefficients from these equations are then hard-coded in the EViews 
*program '2_crisis_probability.prg'.
*Parts of the code taken from Schularick and Taylor's code in FinalResults.do.
********************************************************************************

set more off, perm
set max_memory .
set matsize 8000

*Load dataset
use panel17.dta, clear

*Import credit-to-GDP gap data
use panel17.dta, clear
merge 1:1 iso year using creditgap.dta
drop _merge

*Import updated data for Australia
merge 1:1 year iso using updated_aus_data.dta, nogen
replace loans1 = loans_aus if iso == "AUS"
replace cpi = cpi_aus if iso == "AUS"
replace gdp = gdp_aus if iso == "AUS"
replace crisisST = 0 if iso == "AUS" & year > 2008

*Do not use noisy french loans data before 1895
replace loans1 = . if year<=1895 & iso=="FRA"

*Make ccodes, sort, tsset
egen ccode = group(iso)
tab iso, gen(ccode)
sort ccode year
xtset ccode year

*Make some series
gen lloans = log(loans1)	
gen dlloans = d.lloans
gen loansgdp = loans1/gdp

gen lpc = log(cpi)
gen dlpc = d.lpc

gen lloansr = lloans-lpc
gen dlloansr = dlloans - dlpc
gen dlloansr2 = dlloansr^2

*Generate dummies
gen dum_int_sq = 1
replace dum_int_sq = 0 if dlloansr < 0
gen dlloansr_sq = dlloansr^2*dum_int_sq

gen dum_int = 1
replace dum_int = 0 if dlloansr < 0 & credgdp_gap  < 0
gen interaction = dlloansr*credgdp_gap*dum_int

gen ww1 = 1 if year>=1914 & year<=1919
gen ww2 = 1 if year>=1939 & year<=1947
replace ww1 = 0 if ww1~=1
replace ww2 = 0 if ww2~=1

gen war=1 if ww1==1|ww2==1
replace war=0 if war~=1

*Label main vars
label var dlpc  "D log CPI"
label var dlloans  "D log Loans"
label var dlloansr "D log Real Loans"

*Drop wars
sum year if ww1==1
sum year if ww2==1
drop if ww1==1 | ww2==1

********************************************************************************
*Fixed effects estimates
********************************************************************************

sort ccode year
xtset ccode year 

*Define estimation sample
gen estsamp = (year<=2008)

*Probability of financial crisis in all countries and confidence interval
cii 1946 79, wald
*Alternative methods of calculating confidence intervals
cii 1946 79, exact
cii 1946 79, wilson
cii 1946 79, jeffreys
cii 1946 79, agresti

*Estimate fixed effects logit model
*This is the default model used in Saunders and Tulip (2019)
logit crisisST l(1/5).dlloansr ccode?* if estsamp==1,  vce(robust)
estimates store fe_logit_default

*Predict probability of a financial crisis in Australia
predict plog_fe_prob if year <= 2018, pr
predict plog_fe_ind if year <= 2018, xb
predict plog_fe_indse if year <= 2018, stdp

*Generate confidence intervals
gen plog_fe_ind_lb = plog_fe_ind + invnormal(0.025)*plog_fe_indse
gen plog_fe_ind_ub = plog_fe_ind + invnormal(0.975)*plog_fe_indse
gen plog_fe_prob_lb = exp(plog_fe_ind_lb)/(1+exp(plog_fe_ind_lb))*100
gen plog_fe_prob_ub = exp(plog_fe_ind_ub)/(1+exp(plog_fe_ind_ub))*100
replace plog_fe_prob = plog_fe_prob*100

*Label estimates
label var plog_fe_prob  	"Estimated probability - fixed effects logit"
label var plog_fe_prob_lb  	"Lower - 95% confidence interval"
label var plog_fe_prob_ub  	"Upper - 95% confidence interval"

*Estimate fixed effects linear probability model with quadratic 
*Discussed in Section 5.1.1 in Saunders and Tulip (2019)
reg crisisST l(1/5).dlloansr l(1/5).dlloansr_sq ccode?* if estsamp==1, /// 
vce(robust)
estimates store fe_quadratic
testparm l(1/5).dlloansr_sq

*Estimate fixed effects linear probability model with credit-to-GDP gap
*Discussed in Section 5.1.2 in Saunders and Tulip (2019)
reg crisisST l(1/5).dlloansr l2.credgdp_gap l2.interaction ccode?* if /// 
estsamp==1, vce(robust)
estimates store fe_credit_gap
testparm l2.credgdp_gap
testparm l2.interaction


*******************************************************************************
*Pooled estimates (no FE)
*These estimates are not directlt used or discussed in Saunders and Tulip (2019)
*******************************************************************************

*Pooled logit model
logit crisisST l(1/5).dlloansr if estsamp==1,  vce(robust)
estimates store pool_logit

*Predict probability of a financial crisis in Australia
predict plog_pool_prob if year <= 2018, pr
predict plog_pool_ind if year <= 2018, xb
predict plog_pool_indse if year <= 2018, stdp

*Generate confidence intervals and convert to %
gen plog_pool_ind_lb = plog_pool_ind + invnormal(0.025)*plog_pool_indse
gen plog_pool_ind_ub = plog_pool_ind + invnormal(0.975)*plog_pool_indse
gen plog_pool_prob_lb = exp(plog_pool_ind_lb)/(1+exp(plog_pool_ind_lb))*100
gen plog_pool_prob_ub = exp(plog_pool_ind_ub)/(1+exp(plog_pool_ind_ub))*100
replace plog_pool_prob = plog_pool_prob*100

*Label estimates
label var plog_pool_prob  	"Estimated probability - pooled logit"
label var plog_pool_prob_lb "Lower - 95% confidence interval"
label var plog_pool_prob_ub "Upper - 95% confidence interval"

*Estimate pooled linear probability model with quadratic 
reg crisisST l(1/5).dlloansr l(1/5).dlloansr_sq if estsamp==1, vce(robust)
estimates store pool_quadratic
testparm l(1/5).dlloansr_sq

*Estimate pooled linear probability model with credit-to-GDP gap
reg crisisST l(1/5).dlloansr l2.credgdp_gap l2.interaction if estsamp==1, /// 
vce(robust)
estimates store pool_credit_gap
testparm l2.credgdp_gap
testparm l2.interaction

*******************************************************************************
*Graph credit-to-GDP gap and predicted probabilities for Australia
*******************************************************************************

cd "`savepath'\Graphs"

graph twoway line credgdp_gap year if iso == "AUS" ///
, title("Credit-to-GDP Gap") ///
ytitle("ppt") xtitle("Year") legend(col(1)) xlabel(1868(20)2018)  
graph export "Credit-to-GDP Gap.png", replace
	
graph twoway line plog_fe_prob plog_fe_prob_lb plog_fe_prob_ub year ///
if iso == "AUS", title("Probability of a Financial Crisis in Australia") ///
ytitle("%") xtitle("Year") subtitle("Fixed effects logit") legend(col(1)) /// 
xlabel(1888(20)2018)
graph export "Probability of Crisis (Fixed Effects Logit).png", replace

graph twoway line plog_pool_prob plog_pool_prob_lb plog_pool_prob_ub year ///
if iso == "AUS", title("Probability of a Financial Crisis in Australia") ///
ytitle("%") xtitle("Year") subtitle("Pooled logit") legend(col(1)) ///
xlabel(1888(20)2018)
graph export "Probability of Crisis (Pooled Logit).png", replace

graph twoway line plog_fe_prob plog_pool_prob year ///
if iso == "AUS" & year > 1889 & year <= 2016 ///
, title("Probability of a Financial Crisis in Australia") ///
ytitle("%") xtitle("Year") legend(col(1)) xlabel(1888(20)2018) 
graph export "Probability of Crisis.png", replace

*Export predicted probabilities to Excel
export excel year plog_fe_prob plog_fe_prob_lb plog_fe_prob_ub  /// 
plog_pool_prob plog_pool_prob_lb plog_pool_prob_ub using "StataOutput.xlsx" ///
if iso == "AUS", sheet("crisis_probability") cell("A6") /// 
sheetmodify firstrow(variables)

cd "`savepath'"

capture log close
