##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Collect RBA Forecasts
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file reads in the RBA forecasts
###############################################################################


##### Functions for data import and manipulation ##############################
### Data import
read_fc <- function(var_name, data_dir, file_name, sheet_name = "Forecasts", 
                    skip_num = 1) {
  
  # Define variable name for nowcast
  new_nameq0 <- paste0("fc", var_name, "q0")
  
  ## Read excel
  y <- read_excel(paste0(data_dir, file_name), 
                  sheet = sheet_name, skip = skip_num)
  
  ## Manipulation
  y <- y %>%
    rename(Date = `Forecast date`) %>%
    select(c(Date, t, contains("+"))) %>%
    rename_at(vars(starts_with("t")),  
              list( ~ str_replace(., "t", paste0("fc",var_name)))) %>%
    rename_at(vars(contains("+")), list( ~ str_replace(., "\\+", "q"))) %>%
    rename(!!new_nameq0 := paste0("fc", var_name))
  
  return(y)
}

### Compute Revisions
fc_rev <- function(fcdata) {
  var_name <- deparse(substitute(fcdata))
  fcrev <- fcdata
  lagdata <- fcdata %>%
    mutate_at(vars(starts_with("fc")), list(~ lag(.,1)))
  fcrev[, 2:(ncol(fcrev)-1)] <- fcrev[, 2:(ncol(fcrev)-1)]-lagdata[, 3:(ncol(fcrev))]
  fcrev[, ncol(fcrev)] <- NULL
  fcrev <- rename_at(fcrev, vars(starts_with("fc")), 
                  list( ~ str_replace(., paste0("fc", var_name), paste0("fcr",var_name))))
  
  return(fcrev)
}


##### Step 1: Load Forecast Data ##############################################
### Underlying inflation
cpii <- read_fc("cpii", data_dir, "underlying-inflation-by-horizon.xls", "Forecasts", 1)

### Unemployment
ur <- read_fc("ur", data_dir, "unemployment-rate-by-horizon.xls", "Forecasts", 1)

### GDP
gdp <- read_fc("gdp", data_dir, "gdp-ye-by-horizon.xls", "Forecasts", 2)
# Adjust forecast dates for 2000Q2-2007Q4
gdp <- gdp %>%
  mutate(Date = if_else((Date>as.Date("2000-03-31") & Date<=as.Date("2007-12-31")),
                        Date %m-% months(1), 
                        Date))


##### Step 2: Compute forecasted changes in the unemployement rate ############
### Forecasted change in unemployment rate
dur <- ur %>%
  mutate_at(vars(starts_with("fc")), list(~ .-ur$fcurq0)) %>%
  select(-(fcurq0)) %>%
  rename_at(vars(starts_with("fc")), 
          list(~ str_replace(., "fc", "fcd")))


##### Step 3: Compute forecast revisions ######################################
### Inflation
rcpii <- fc_rev(cpii)

### Unemployment
rur <- fc_rev(ur)

### Change in Unemployment
rdur <- fc_rev(dur)

### GDP
rgdp <- fc_rev(gdp)


##### Step 4: Merge all data frames ###########################################
fc_list <- list(cpii, ur, gdp, dur, rcpii, rur, rgdp, rdur)
fc_data <- join_all(fc_list, by = "Date", type = "full")
fc_data <- fc_data %>% 
  mutate(Date = as_date(Date)) %>% 
  as_tibble()


##### Step 5: Change date to date of Board Meeting ############################
cr_dates <- cr %>% 
  select(Date) %>% 
  rename(BoardDate = Date) %>% 
  mutate(Date = floor_date(BoardDate, "month"))

fc_data <- left_join(fc_data, cr_dates, by = "Date")
fc_data <- fc_data %>% 
  mutate(Date = BoardDate) %>% 
  select(-(BoardDate)) %>% 
  filter_if(is.numeric, any_vars(!is.na(.)))

fc_data <- mutate(fc_data, fc_mth = 1)

fc_names <- names(fc_data)

##### Step 6: Save data #######################################################
rm(cpii, ur, gdp, dur, rcpii, rur, rgdp, rdur)
save(fc_data, file = file.path(data_dir, "fcdata.RData"))
