##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: RBA Inflation Forecast Errors: The Role of Credit Spreads - Subsample
#       Evidence
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file estimates to which extent money and credit market 
#              spreads and explain the Bank's inflation forecast errors four 
#              four different subsamples.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Table 8: Credit Market Conditions and 1-Year Ahead Inflation 
#             Forecast Errors - Subsample Evidence
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Load functions
source("Code/functions/funcs.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer", "plyr",
                          'vars', "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
horizon <- 4
p_ar <- 4
p_sv <- 1
p_cs <- 1
p_crexp <- 1

## Forecast Variables 
dep_names <- c("cpii")
dep_transform <- c("lnd")
dep_vars <- data.frame(dep_names, dep_transform, stringsAsFactors=FALSE)
colnames(dep_vars) <- c("variable", "transform")
dep_vars <- mutate(dep_vars, tf_variable = ifelse(transform=="l", variable,
                                                  paste0(transform, horizon, "_", variable)))

## RBA forecasts and revisions
fcast_vars <- paste0("fc", dep_names, "q", horizon)

## State variables
# "d" for level change, "lnd" for growth rate
sv_names <- c("cpii", "ur", "gdp")
sv_transform <- c("lnd", "d", "lnd")
state_vars <- data.frame(sv_names, sv_transform, stringsAsFactors=FALSE)
if (nrow(state_vars)>0) {
  colnames(state_vars) <- c("variable", "transform")
  state_vars <- mutate(state_vars, tf_variable = ifelse(transform=="l", variable,
                                                        paste(transform, variable, sep="_")))
}

## Cash rate expectations (i.e. yield curve)
cr_exp <- "sprd_yc_ois3"

## Credit spreads and risk measures
cs_vars <- c("sprd_ir_bab_ois3", "sprd_ir_lbiz_bab3", "sprd_ir_us_cbb_gb10", "us_vix")
cs_vars_main <- c(1,2)

### Sample
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")
# Sample omission (specify start and end date)
smpl_omit <- c(as.Date("2008-06-01"), as.Date("2009-06-01"))


##### Step 4: Load and clean data and construct missing variables #############
## Load data
load(paste0(data_dir, "data_final.RData"))

## Select variables from dataset
Y <- data_mthly %>% 
  select(Date, fc_mth, !!fcast_vars, !!dep_names, !!sv_names, !!cr_exp, !!cs_vars)

## Transformations
y_sv <- select(Y, Date, !!sv_names) %>%  
  transform_df(state_vars, l=3)
y_dep <- select(Y, Date, !!dep_names) %>%  
  transform_df(dep_vars, l=3*horizon)

## Combine data
Y <- select(Y, Date, fc_mth, !!fcast_vars, !!cr_exp, !!cs_vars) %>% 
  left_join(y_sv, by="Date") %>% 
  left_join(y_dep, by="Date")

## Compute forecast errors
for (j in 1:nrow(dep_vars)) {
  err_name <- paste0("err_", dep_vars$variable[j])
  Y <- mutate(Y, !!err_name := lead(!!as.name(dep_vars$tf_variable[j]), 3*horizon) - !!as.name(fcast_vars[j]))
}

## Account for real-time availability of financial and real-economic data
Y <- Y %>% 
  mutate_at(vars(cr_exp, cs_vars), lag) 
Y <- Y %>%
  mutate_at(vars(state_vars$tf_variable), list(~lag(.,2)))

## Keep forecast months only
Y <- filter(Y, fc_mth==1) %>% 
  select(-fc_mth)

### Add GFC Dummy
Y <- mutate(Y, gfc_dum = 1*(Date>=smpl_omit[1] & Date<=smpl_omit[2]))

### Keep maximum permissible sample
Y <- filter(Y, Date>=smpl_start & Date<=smpl_end)


##### Step 5: Regressions - Forecast Efficiency Tests #########################
modnum <- 0     # set model number
adl_error <- list()
adl_se <- list()
for (spec in 1:4) {
  if (spec==1) {
    cs_vars <- c(cs_vars[cs_vars_main], "gfc_dum")
    Y_sub <- Y
  } else if (spec==2) {
    cs_vars <- cs_vars[cs_vars_main]
    Y_sub <- Y %>% 
      mutate_at(vars(err_name), list(~ ifelse((Date>=smpl_omit[1] & Date<=smpl_omit[2]),NA,!!as.name(err_name))))
  } else if (spec==3) {
    cs_vars <- cs_vars[cs_vars_main]
    Y_sub <- filter(Y, (Date<smpl_omit[1]))
  } else if (spec==4) {
    cs_vars <- cs_vars[cs_vars_main]
    Y_sub <- filter(Y, (Date>smpl_omit[2]))
  }
  
  yx.zoo <- select(Y_sub, -Date)
  yx.zoo <- zoo(yx.zoo, Y_sub$Date)
  
  
  # Benchmark (only controls)
  modnum <- modnum+1
  x_sv <- paste0(err_name, "~ L(", err_name, ", ", horizon, ":", horizon+p_ar-1, ")")
  for (j in 1:nrow(state_vars)) {
    x_sv <- paste0(x_sv, " + L(", state_vars$tf_variable[j], ", 0:", p_sv-1, ")")
  }
  adl_error[[modnum]] <- dynlm(as.formula(x_sv), data = yx.zoo)
  adl_se[[modnum]] <- coeftest(adl_error[[modnum]], vcov. = NeweyWest(adl_error[[modnum]], prewhite = F))[,2]
  
  # Joint credit spread variables
  modnum <- modnum+1
  x_csj <- x_sv
  for (k in 1:length(cs_vars)) {
    x_csj <- paste0(x_csj, " + L(", cs_vars[k], ", 0:", as.character(p_cs-1), ")")
  }
  adl_error[[modnum]] <- dynlm(as.formula(x_csj), data = yx.zoo)
  adl_se[[modnum]] <- coeftest(adl_error[[modnum]], vcov. = NeweyWest(adl_error[[modnum]], prewhite = F))[,2]
}

##### Step 6: Show Results ####################################################
collab <- c("GFC Dummy", "Excl. GFC", "Pre-GFC", "Post-GFC")
stargazer(adl_error[seq(2,modnum,2)], se=adl_se, 
          title = "Credit Market Conditions and 1-Year Ahead Inflation Forecast Errors", 
          align = TRUE, type = "text", no.space = TRUE, dep_var.caption = "", 
          column.labels = collab, model.numbers = FALSE,
          omit = "err", omit.labels = "AR Terms", 
          omit.stat = c("ser"), digits = 3, df = FALSE)
for (n in 1:spec) {
  print(anova(adl_error[[2*n-1]], adl_error[[2*n]]))
}
