##### Preamble ################################################################s
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Monetary Policy Effects from Local Projections: Bishop and Tulip (2017)
#       policy shock vs BT shock purged of response to creadit spreads (BT-CS 
#       shock)
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file estimates the effect of a cash rate change using Local
#              Projections to create Figure D2 of the paper.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Figure D2 (for RBA users only, PNG-format)
#          2. Data for Figure D2 in Excel-file.
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Load functions
source("Code/functions/funcs.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer", "plyr",
                          'vars', "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
horizon <- 20     # Impulse horizon (in quarters)
cumirf <- 1       # Compute cumulative impulse responses? 1: Yes
alpha <- 0.90     # Confidence interval

## Monetary policy shock series
shocks <- c("rr_fw", "rr_fw_csj_unant")

## Dependent Variables and Policy Shocks
dep_names <- c("cpii", "ur")
dep_transform <- c("lnd", "d")
dep_vars <- data.frame(dep_names, dep_transform, stringsAsFactors=FALSE)
colnames(dep_vars) <- c("variable", "transform")
dep_vars <- mutate(dep_vars, tf_variable = ifelse(transform=="l", variable,
                                                  paste(transform, variable, sep="_")))
l_ar <- 4

## Additional controls in LP regression
# "d" for level change, "lnd" for growth rate, numeric defines change horizon in months
control_names <- c("cpii", "ur", "gdp") # "commpi", "tot", "usffr", "rtwi"
control_transform <- c("lnd", "d", "lnd")
control_vars <- data.frame(control_names, control_transform, stringsAsFactors=FALSE)
colnames(control_vars) <- c("variable", "transform")
control_vars <- mutate(control_vars, tf_variable = ifelse(transform=="l", variable,
                                                  paste(transform, variable, sep="_")))
l_c <- 1

## Lag lengths
l_shk <- 4       # Lag length of shock included
h0x <- 0          # Minimum lag with which policy shock affects dependent variable 
                  # 0: allow contemporaneous effect; 
                  # 1: impose 1 period lag until effect
cholx <- h0x+1    # Assume contemporaneous effect of policy on controls? 
                  # 0: Cholesky assumption imposed, monetary policy shock assumed not to affect controls contemporaneously
                  # 1: no Cholesky assumption, policy shock could affect controls contemporanously, take lag of controls

## Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")
# Sample omission (specify start and end date)
smpl_omit <- c()  # as.Date("2008-06-01"), as.Date("2009-06-01")


##### Step 4: Load and clean data and construct missing variables #############
### Policy Shocks
load(paste0(results_dir, "rr_shocks.RData"))
if (any(str_detect(shocks, "unant"))) {
  load(paste0(results_dir, "rr_shocks_unant.RData"))
  rr_shocks <- left_join(rr_shocks, rr_shocks_unant, by="Date")
}
# Select shocks
rr_shocks <- select(rr_shocks, Date, !!shocks)

### VAR data
load(paste0(data_dir, "data_final.RData"))

# Select variables from dataset
Y <- data_mthly %>% 
  select(Date, !!dep_names, !!control_names, fc_mth)

# Keep forecast months only
Y <- filter(Y, fc_mth==1) %>% 
  select(-fc_mth)

### Merge shocks and VAR data
Y <- left_join(Y, rr_shocks, by="Date")

# Adjust date to end of quarter for Board Meetings since 2000:Q2
Y <- mutate_at(Y, vars(Date), list(~ ifelse(.>as.Date("2000-03-31", format = "%Y-%m-%d"),
                                            Date %m+% months(1), .))) %>% 
  mutate(Date = as.Date(Date, format = "%Y-%m-%d", origin = "1970-01-01"))


##### Step 5: Local Projection regressions ####################################
for (j in 1:nrow(dep_vars)) {
  dep_var <- dep_vars[j,]
  for (s in 1:length(shocks)) {
    # Select shock
    mp_shk <- shocks[s]
    
    IRF <- lp_irf(Y, horizon, dep_var, mp_shk, control_vars, smpl_start, smpl_end, 
                  l_ar, l_shk, l_c, h0x, cholx, alpha)
    
    # Transform to long format and re-order columns by variable names
    IRFs_lg <- pivot_longer(IRF, -horizon, names_to = "variable", values_to = "value") %>% 
      arrange(variable, horizon) %>% 
      mutate(shock = mp_shk)
    
    if (j==1 & s==1) {
      IRFs_all_lg <- IRFs_lg
    } else {
      IRFs_all_lg <- bind_rows(IRFs_all_lg, IRFs_lg)
    }
  }
}
IRFs_all <- pivot_wider(IRFs_all_lg, names_from = c(variable, shock), names_sep = "_", 
                        values_from = value)

IRFs_all_lg <- IRFs_all %>% 
  select(matches('horizon|cpii|ur')) %>% 
  pivot_longer(-horizon, names_to = "variable", values_to = "value") %>% 
  arrange(variable, horizon)

##### Step 6: Plot IRFs #######################################################
### CPI and UR
if ("arphit" %in% installed.packages()[,"Package"]) {
  figD2 <- arphitgg(aes = agg_aes(x = horizon, y = value, group = variable), layout = "2b2"
                    , dropxlabel = TRUE, showallxlabels = FALSE) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'cpii') & str_detect(variable, 'rr_fw_csj', negate=T)), panel = "1",
             colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Red6"]), lty = c(2,1,2), lwd = rep(3,3)) +
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'cpii') & str_detect(variable, 'rr_fw_csj')), panel = "2",
             colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Red6"]), lty = c(2,1,2), lwd = rep(3,3)) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'ur') & str_detect(variable, 'rr_fw_csj', negate=T)), panel = "3",
             colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Red6"]), lty = c(2,1,2), lwd = rep(3,3)) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'ur') & str_detect(variable, 'rr_fw_csj')), panel = "4",
             colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Red6"]), lty = c(2,1,2), lwd = rep(3,3)) + 
    agg_yaxislabel("CPI", panel = "1") + 
    agg_yaxislabel("Unemployment", panel = "3") + 
    agg_title("BT Shock", panel = "1") +
    agg_title("Unanticipated BT-CS Shock", panel = "2") +
    agg_units("ppt", panel = c("3","4")) +
    agg_ylim(-5, 5, 5, panel = c("1","2")) +
    agg_ylim(-2, 2, 5, panel = c("3","4")) +
    agg_title("Impulse Responses from Local Projections") +
    agg_subtitle("Cumulative quarterly response to 100 bps policy shock; 1994:Q1-2018:Q4*") +
    agg_source("ABS") +
    agg_source("Author's calculations") +
    agg_source("RBA") + 
    agg_footnote("Cumulative quarterly response to 100 bps cash rate shock. Left panel shows response to original, anticipated Bishop and Tulip (2017) shock. Right panel shows responses to new preferred unanticipated shock. Dashed lines show 90% heteroscedasticity and autocorrelation robust confidence bands.")
  agg_draw(figD2, filename = paste0(results_dir, "FigD2_LP_IRFs.png"))
}

### Export graph data
IRFs_lp <- mutate(IRFs_all_lg, panel = case_when((variable=='cpii_rr_fw') ~ 1, 
                                                 (variable=='cpii_lwr_rr_fw') ~ 1, 
                                                 (variable=='cpii_upr_rr_fw') ~ 1, 
                                                 (variable=='cpii_rr_fw_csj_unant') ~ 2, 
                                                 (variable=='cpii_lwr_rr_fw_csj_unant') ~ 2, 
                                                 (variable=='cpii_upr_rr_fw_csj_unant') ~ 2, 
                                                 (variable=='ur_rr_fw') ~ 3, 
                                                 (variable=='ur_lwr_rr_fw') ~ 3, 
                                                 (variable=='ur_upr_rr_fw') ~ 3, 
                                                 (variable=='ur_rr_fw_csj_unant') ~ 4, 
                                                 (variable=='ur_lwr_rr_fw_csj_unant') ~ 4, 
                                                 TRUE ~ 4)) %>% 
  arrange(panel, variable, horizon) %>% 
  mutate(label = case_when((variable=='cpii_rr_fw') ~ "CPI (BT Shock)", 
                           (variable=='cpii_lwr_rr_fw') ~ "CPI Lower (BT Shock)", 
                           (variable=='cpii_upr_rr_fw') ~ "CPI Upper (BT Shock)", 
                           (variable=='cpii_rr_fw_csj_unant') ~ "CPI (BT-CS Shock)", 
                           (variable=='cpii_lwr_rr_fw_csj_unant') ~ "CPI Lower (BT-CS Shock)", 
                           (variable=='cpii_upr_rr_fw_csj_unant') ~ "CPI Upper (BT-CS Shock)", 
                           (variable=='ur_rr_fw') ~ "UR (BT Shock)", 
                           (variable=='ur_lwr_rr_fw') ~ "UR Lower (BT Shock)", 
                           (variable=='ur_upr_rr_fw') ~ "UR Upper (BT Shock)", 
                           (variable=='ur_rr_fw_csj_unant') ~ "UR (BT-CS Shock)", 
                           (variable=='ur_lwr_rr_fw_csj_unant') ~ "UR Lower (BT-CS Shock)", 
                           TRUE ~ "UR Upper (BT-CS Shock)")) %>% 
  select(-c(variable, panel)) %>% 
  pivot_wider(names_from = label, values_from = value)
write_xlsx(IRFs_lp, path = paste(results_dir, "FigD2_LP_IRFs.xlsx", sep=""), col_names = TRUE)
