##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Collect Data
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file collects all necessary data from various Bank and 
#              online data sources, and exports the data in an Rdata file. As 
#              most data is not directly accessible for non-Bank staff, only 
#              the final dataset will be shared but not the file itself.
# User inputs:  1. Set directories in Step 1
# Outputs:      1. Rdata file with dataset.
###############################################################################


##### Step 1: Set Directories and select sample start #########################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"

### Sample 
# Start
smplstart <- as.Date("1986-01-01")
# Maximum
smplmax <- as.Date("2020-12-31")

### Get functions
source("Code/functions/funcs.R")

### Load necessary packages
check_and_install_packs(c("lubridate", "zoo", "rvest", "httr", "plyr", "quantmod",
                          "readxl", "tidyverse"))

### Load RBAverse
rbaverse::update()

### Load Gertrude package
pkg_rGertrude <- paste(tempdir(), 'rGertrude_0.2.1.zip', sep = '\\')
download.file('https://trimweb.rba.gov.au/record/D18~334439', pkg_rGertrude, mode ='wb')
install.packages(pkg_rGertrude, repos = NULL)

library(rGertrude) 
rGertrude::load_cookie()


##### Step 2: Load Data #######################################################
### Read excel spreadsheet with data information
metadata <- get_metadata(data_dir)


### Cash rate changes
source('Code/01a_cash_rate_changes.r')


### RBA Forecasts
source('Code/01b_rba_fcasts.r')


### Australian macro data from GERTRUDE
GERT.zoo <- xlgertrude_timeseries(metadata[!is.na(metadata$`Gertrude Series Code`), ])
gert <- data.frame(index(GERT.zoo), as.data.frame(GERT.zoo))
gert <- gert %>% 
  rename(Date = index.GERT.zoo.) %>% 
  filter(Date>=smplstart) %>% 
  as_tibble()
gert_names <- names(gert)


### US Macro Data from FRED
source('Code/01c_fred_data.r')


##### Collect Australian interest rates data ##################################
source('Code/01d_aus_ir.R')


##### Collect Money market and zero-coupon forward rates ######################
source('Code/01e_money_market.R')

# Update last available date
smplmax <- max(mm_rates$Date)


##### Step 3: Merge all the data series #######################################
data_list <- list(fc_data, cr, gert, fred, gov_bond, ir_lend, corp_bond, mm_rates)
data <- join_all(data_list, by = "Date", type = "full")
data <- data %>% 
  arrange(Date) %>% 
  mutate(Date = as_date(Date)) %>% 
  filter(Date<=smplmax) %>% 
  as_tibble()


##### Step 4: Construct further variables #####################################
### Calculate interest rate spreads
data <- data %>% 
  mutate(MonthDate = floor_date(Date, unit="months"),
         # Money Market Yield curve
         sprd_yc_ois3 = ir_ois3 - ir_zcf0,
         sprd_yc_zcf3 = ir_zcf3 - ir_zcf0,
         # Money Market Risk spreads
         sprd_ir_bab_ois1 = ir_bab1 - ir_ois1,
         sprd_ir_bab_ois3 = ir_bab3 - ir_ois3,
         sprd_ir_bab_ois6 = ir_bab6 - ir_ois6,
         # AUS Bond Spreads
         sprd_ir_cb5 = ir_cbb5 - ir_cba5,
         sprd_ir_cb10 = ir_cbb10 - ir_cba10,
         sprd_ir_cbb_gb5 = ir_cbb5 - ir_gb5,
         sprd_ir_cbb_gb10 = ir_cbb10 - ir_gb10,
         # US Bond Spreads
         sprd_ir_us_cbb_gb10 = us_ir_cbb - us_ir_gb10)


##### Step 5: Create monthly series ###########################################
### Get end of month values for all observations
data_mthly <- data %>% 
  group_by(MonthDate) %>%
  mutate_at(vars(-MonthDate), list(~ na.locf(., na.rm = FALSE))) %>% 
  ungroup() %>% 
  filter(Date==(ceiling_date(Date, unit="months")-days(1))) %>% 
  mutate(Date = MonthDate) %>% 
  select(-MonthDate)

### Fill in January observations for cash rate
data_mthly <- data_mthly %>% 
  mutate(cr = if_else(is.na(cr), lag(cr), cr),
         cr_chg = if_else(is.na(cr_chg), 0, cr_chg),
         l.cr = if_else(is.na(l.cr), lag(cr), l.cr))

### Replace NA in Forecast Months by Zeros
data_mthly <- mutate(data_mthly, fc_mth = ifelse(is.na(fc_mth), 0, fc_mth))

### Construct Household Business Credit Spreads
data_mthly <- mutate(data_mthly, sprd_ir_mrg_cr = ir_mrg - cr,
                     sprd_ir_lbiz_bab3 = ir_lbiz - ir_bab3,
                     sprd_ir_sbiz_lbiz = ir_sbiz - ir_sbiz)


### Roll all quarterly observations forward except forecast revisions
data_mthly <- mutate_at(data_mthly, vars(-starts_with("fcr")), 
                        list(~ na.locf(., na.rm = FALSE)))


##### Step 6: Variable Transformations ########################################
### Deflate selected series
deflate_cols <- filter(metadata, deflate == 1) %>% 
  select(Mnemonic) %>% 
  as.character()
data_mthly <- mutate_at(data_mthly, vars(starts_with(deflate_cols)), 
                        list(~ deflate_ser(., cpi)))

##### Step 7: Save Data in R-data file ########################################
save(list = c("data_mthly", "fc_names", "gert_names", "fred_names", 
              "gov_bond_names", "ir_lend_names", "corp_bond_names", "mm_rates_names"), 
     file = file.path(data_dir, "data_full_current.RData"))

