##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Plot data - Figures 2, 3, and 4
# Creator: Ben Beckers
# Date: 22 January 2020
# Description: This file collects and exports the Bank's forecasts and measures
#              of credit spreads and uncertainty shown in Figures 2 to 4.
#              For RBA users, the file also creates and exports the png-files.
# User inputs: 1. Set directories in Step 1
#              2. Select sample in Step 2
#              3. Select variables for figures in Step 2.
# Outputs: 1. Correlation matrices between measures of credit spreads and 
#             uncertainty.
#          2. Figures 2, 3 and 4 (for RBA users only, PNG-format)
#          3. Data for Figures 2, 3, and 4 in Excel-files.
###############################################################################


##### Step 1: Set Directories, load functions and packages ####################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"

# Source functions
source("Code/functions/funcs.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("lubridate", "zoo", "rvest", "httr", "plyr", "quantmod",
                          "arphit", "writexl", "readxl", "tidyverse"))


### Step 2: Select sample and variables #######################################
### Select sample
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2019-01-01")
# Subsample for Figure 2
fig2_start <- as.Date("2007-01-01")
fig2_end <- as.Date("2011-12-31")
# Subsample for Figure 3
fig3_start <- as.Date("2006-01-01")
fig3_end <- as.Date("2011-12-31")
# SMP forecast months for forecasts shown in Figure 4
fcast_dates <- as.Date(c("2008-08-01", "2008-11-01", "2009-02-01", "2009-05-01"))

### Select Variables
# Credit spreads und uncertainty measures
cs_vars <- c("sprd_ir_cbb_gb10", "sprd_ir_bab_ois3", "sprd_ir_lbiz_bab3", 
             "sprd_ir_us_cbb_gb10", "us_vix")

# Cash rate expectations
dcr_exp <- paste0("sprd_yc_ois3")  # Expected cash rate change over next quarter

# SMP forecasts
horizon <- seq(0,10,1)
fc_cpii <- paste0("fccpiiq", as.character(horizon))
fc_ur <- paste0("fcurq", as.character(horizon))


### Step 3: Load data and construct missing variables #########################
# Load data
load(paste0(data_dir, "data_final.Rdata"))

write_xlsx(data_mthly, path = paste(data_dir, "data_final.xlsx", sep=""), col_names = TRUE)


# Select sample
data_mthly <- filter(data_mthly, Date>=smpl_start & Date<smpl_end)


##### Step 4: Correlations between Credit Spreads #############################
### Monthly levels
# Common sample (for which AUS corporate bonds are available)
select(data_mthly, c(cs_vars, dcr_exp)) %>% 
  cor(use="complete.obs")

### Monthly differences
# Common sample
select(data_mthly, c(cs_vars, dcr_exp)) %>% 
  mutate_at(vars(contains("sprd")), diff2df) %>% 
  mutate_at(vars(contains("us_vix")), logdiff2df) %>% 
  cor(use="complete.obs")


##### Step 5: Graph data ######################################################

### Figure 2: Credit Spreads and Uncertainty during the GFC
data_fig2 <- filter(data_mthly, Date>=fig2_start & Date<=fig2_end) %>% 
  select(Date, !!cs_vars[-3])

dl_fig2 <- pivot_longer(data_fig2, -Date, names_to = "variable", values_to = "value") %>% 
  arrange(variable, Date)
if ("arphit" %in% installed.packages()[,"Package"]) {
  fig2 <- arphitgg(aes = agg_aes(x = Date, y = value, group = variable)) + 
    agg_line(data = dl_fig2 %>% filter(variable != "us_vix"), panel = "1",
             colour = c(RBA["Green3"], RBA["DarkGreen7"], RBA["Red6"])) +
    agg_line(data = dl_fig2 %>% filter(variable == "us_vix"), panel = "2",
             colour = RBA["Purple6"]) +
    agg_units("%", panel = "1") +
    agg_units("Index", panel = "2") +
    agg_ylim(0,10,5,panel="1") +
    agg_ylim(0,80,5,panel="2") + 
    agg_xlim(year(fig2_start), year(fig2_end)+1) + 
    agg_title("Credit Spreads and Uncertainty during the GFC") +
    agg_subtitle("2007:M1 - 2011:M12") +
    agg_source("Federal Reserve Bank of St. Louis") +
    agg_source("RBA") + 
    agg_label("AUS BBB Corporate Bond**", x = 2010.5, y = 8.5, colour = RBA["DarkGreen7"], panel = "1") +
    agg_label("Australian Money Market***", x = 2010.5, y = 1, colour = RBA["Green3"], panel = "1") +
    agg_label("US BAA Corporate Bond*", x = 2010.6, y = 5.25, colour = RBA["Red6"], panel = "1") +
    agg_label("US VIX (RHS)****", x = 2007.9, y = 50, colour = RBA["Purple6"], panel = "2") +
    agg_footnote("US BAA Corporate Bond Rate to 10-year US Treasury Bond Yield") + 
    agg_footnote("BBB Corporate Bond Rate to 10-year Australian Government Bond Yield") + 
    agg_footnote("3-month Bank Bill Rate to Zero-coupon Yield") + 
    agg_footnote("Chicago Board Options Exchange, CBOE Volatility Index")
  agg_draw(fig2, filename = paste0(results_dir, "Fig2_Credit_Spreads_GFC.png"))
}
dl_fig2 <- dl_fig2 %>% 
  mutate(label = case_when(str_detect(variable, "sprd_ir_cbb_gb10") ~ "AUS BBB Corporate Bond", 
                           str_detect(variable, "sprd_ir_bab_ois3") ~ "Australian Money Market", 
                           str_detect(variable, "sprd_ir_us_cbb_gb10") ~ "US BAA Corporate Bond", 
                           TRUE ~ "US VIX (RHS)"))
data_fig2 <- select(dl_fig2, -variable) %>% 
  pivot_wider(id_cols = Date, names_from = label, values_from = value)
write_xlsx(data_fig2, path = paste(results_dir, "Fig2_Credit_Spreads_GFC.xlsx", sep=""), col_names = TRUE)


### Figure 4: Credit Spreads and Uncertainty - full sample
data_fig4 <- mutate_at(data_mthly, vars(cs_vars), lag) %>% 
  filter(fc_mth==1) %>% 
  select(Date, !!cs_vars)

dl_fig4 <- pivot_longer(data_fig4, -Date, names_to = "variable", values_to = "value") %>% 
  arrange(variable, Date)
if ("arphit" %in% installed.packages()[,"Package"]) {
  fig4 <- arphitgg(aes = agg_aes(x = Date, y = value, group = variable)) + 
    agg_line(data = dl_fig4 %>% filter(variable != "us_vix"), panel = "1",
             colour = c(RBA["Green3"], RBA["DarkGreen7"], RBA["Olive8"], RBA["Red6"])) +
    agg_line(data = dl_fig4 %>% filter(variable == "us_vix"), panel = "2",
             colour = RBA["Purple6"]) +
    agg_units("%", panel = "1") +
    agg_units("Index", panel = "2") +
    agg_ylim(-2,10,7,panel="1") +
    agg_ylim(0,90,7,panel="2") + 
    agg_xlim(year(smpl_start), year(smpl_end)) + 
    agg_title("Measures of Credit Market Conditions and Uncertainty") +
    agg_subtitle("1994:Q1 - 2018:Q4*") +
    agg_source("Federal Reserve Bank of St. Louis") +
    agg_source("RBA") + 
    agg_label("AUS BBB Corporate Bond**", x = 2003, y = 8.5, colour = RBA["DarkGreen7"], panel = "1") +
    agg_label("Australian Money Market**", x = 2001.5, y = -1.2, colour = RBA["Green3"], panel = "1") +
    agg_label("Large Business Lending***", x = 2000.5, y = 5, colour = RBA["Olive8"], panel = "1") +
    agg_label("US BAA Corporate Bond**", x = 2003, y = 6.4, colour = RBA["Red6"], panel = "1") +
    agg_label("US VIX (RHS)**", x = 2011, y = 6, colour = RBA["Purple6"], panel = "2") +
    agg_footnote("As available at the beginning of forecast months (end-of-quarter of months prior to 2000, mid-quarter months since)") + 
    agg_footnote("See Figure 2.") + 
    agg_footnote("Large Business Variable Lending Rate to 3-month Bank Bill Rate")
  agg_draw(fig4, filename = paste0(results_dir, "Fig4_Credit_Spreads_Full.png"))
}
dl_fig4 <- dl_fig4 %>% 
  mutate(label = case_when(str_detect(variable, "sprd_ir_cbb_gb10") ~ "AUS BBB Corporate Bond", 
                           str_detect(variable, "sprd_ir_bab_ois3") ~ "Australian Money Market", 
                           str_detect(variable, "sprd_ir_lbiz_bab3") ~ "Large Business Lending", 
                           str_detect(variable, "sprd_ir_us_cbb_gb10") ~ "US BAA Corporate Bond", 
                           TRUE ~ "US VIX (RHS)"))
data_fig4 <- select(dl_fig4, -variable) %>% 
  pivot_wider(id_cols = Date, names_from = label, values_from = value)
write_xlsx(data_fig4, path = paste(results_dir, "Fig4_Credit_Spreads_Full.xlsx", sep=""), col_names = TRUE)


### Figure 3: Inflation and Unemployment Forecasts during the GFC
data_fig3 <- filter(data_mthly, Date>=fig3_start & Date<=fig3_end & fc_mth==1) %>% 
  select(Date, cpii, ur, !!fc_cpii, !!fc_ur)

# Compute annual inflation
data_fig3 <- data_fig3 %>%
  mutate(cpii = 100*logdiff2df(cpii, l=4))

# Create new forecast variables
for (j in 1:length(fcast_dates)) {
  # Find forecast date
  indexstart <- which((data_fig3$Date==fcast_dates[j]))
  # Find forecasts
  fc_cpii_trash <- filter(data_fig3, Date==fcast_dates[j]) %>% 
    select(!!fc_cpii)
  fc_ur_trash <- filter(data_fig3, Date==fcast_dates[j]) %>% 
    select(!!fc_ur)
  cpii_name <- paste0("fc_cpii_", format(fcast_dates[j], "%Y%m"))
  ur_name <- paste0("fc_ur_", format(fcast_dates[j], "%Y%m"))
  # Create new forecast variables
  data_fig3 <- mutate(data_fig3, !!cpii_name := NA,
                      !!ur_name := NA)
  data_fig3[indexstart:(indexstart+max(horizon)), cpii_name] <- t(fc_cpii_trash)
  data_fig3[indexstart:(indexstart+max(horizon)), ur_name] <- t(fc_ur_trash)
}
rm(indexstart, fc_cpii_trash, fc_ur_trash, cpii_name, ur_name)

data_fig3 <- select(data_fig3, -c(!!fc_cpii, !!fc_ur))

# Long format and plot
dl_fig3 <- data_fig3 %>%
  pivot_longer(-Date, names_to = "variable", values_to = "value") %>%
  mutate(panel = case_when(str_detect(variable, "ur|UR") ~ "Unemployment", TRUE ~ "Inflation"))
if ("arphit" %in% installed.packages()[,"Package"]) {
  fig3 <- arphitgg(aes = agg_aes(x = Date, y = value, group = variable), layout = "2h") + 
    agg_line(data = dl_fig3 %>% filter(panel == "Inflation"), panel = "1",
           colour = c(RBA["Grey5"], RBA["Olive1"], RBA["Orange2"], RBA["Red5"], RBA["Red9"])) +
    agg_line(data = dl_fig3 %>% filter(panel == "Unemployment"), panel = "3",
             colour = c(RBA["Olive1"], RBA["Orange2"], RBA["Red5"], RBA["Red9"], RBA["Grey5"])) + 
    agg_xlim(year(fig3_start)+1, year(fig3_end)+1) +
    agg_title("Inflation", panel = "1") + 
    agg_title("Unemployment", panel = "3") + 
    agg_title("Forecasts and the Cash Rate during the GFC") +
    agg_subtitle("Board Meetings, June 2008 - June 2009*") +
    agg_source("Author's calculations") +
    agg_source("RBA") + 
    agg_label("Actual", x = 2007.75, y = 2.5, colour = RBA["Grey5"], panel = "1") +
    agg_label("Aug 08\n(0 bps)", x = 2008, y = 4.5, colour = RBA["Olive1"], panel = "1") +
    agg_label("Nov 08\n(-200 bps)", x = 2010.5, y = 4, colour = RBA["Orange2"], panel = "1") +
    agg_label("Feb 09\n(-400 bps)", x = 2009.25, y = 2.8, colour = RBA["Red5"], panel = "1") +
    agg_label("May 09\n(-400 bps)", x = 2009, y = 7, colour = RBA["Red9"], panel = "3") +
    agg_footnote("Forecasts of underlying inflation (year-ended) and the unemployment rate as published in the Statement on Monetary Policy (SMP). Cumulative cash rate changes since June 2008 in parentheses.") 
  agg_draw(fig3, filename = paste0(results_dir, "Fig3_Forecasts_GFC.png")) # 
}
dl_fig3 <- dl_fig3 %>% 
  mutate(label = case_when(variable=="cpii" ~ "Inflation (Actual)", 
                           variable=="fc_cpii_200808" ~ "Inflation (Aug 08)", 
                           variable=="fc_cpii_200811" ~ "Inflation (Nov 08)", 
                           variable=="fc_cpii_200902" ~ "Inflation (Feb 09)", 
                           variable=="fc_cpii_200905" ~ "Inflation (May 09)", 
                           variable=="ur" ~ "Unemployment (Actual)", 
                           variable=="fc_ur_200808" ~ "Unemployment (Aug 08)", 
                           variable=="fc_ur_200811" ~ "Unemployment (Nov 08)", 
                           variable=="fc_ur_200902" ~ "Unemployment (Feb 09)",
                           TRUE ~ "Unemployment (May 09)"))
data_fig3 <- select(dl_fig3, -c(variable,panel)) %>% 
  pivot_wider(id_cols = Date, names_from = label, values_from = value)
write_xlsx(data_fig3, path = paste(results_dir, "Fig3_Forecasts_GFC.xlsx", sep=""), col_names = TRUE)
