### Preamble ##################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Romer and Romer Monetary Policy Shocks for Australia:
#       Are RR-Shocks Anticipated?
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file estimates tests if cash rate changes are predictable 
#              by financial market's expectations for cash rate changes.
#              1. Estimate the cash rate 'response' to the Bank's forecasts, a
#                 set of variables (real economic, financial) and the expected
#                 change in the cash rate from financial market data.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Table 2: Regression results from Romer and Romer Regression 
#                      including expected cash rate changes
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Source functions
source("Code/functions/funcs.R")
source("Code/functions/rr_regress.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer",
                          "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
dep_var <- "cr_chg"       # Dependent LHS variable (change in cash rate)
lag_dep_var <- "cr"       # Lagged dependent variable (from cr, cr_chg)
adlp <- 1                 # Number of lags of lagged dependent variable

## RBA forecasts and revisions
fcast_vars <- c("fccpiiq2", "fcgdpq2", "fcurq0", "fcrcpiiq2", "fcrgdpq2")
# "fc" for forecast, "fcr" for forecast revision, "q'h'" for quarterly horizon
# "cpii" for underlying inflation, "gdp" for GDP, "ur" for level of unemployment rate, 
# "dur" for change in unemployment rate

## Cash rate expectations (i.e. yield curve)
cr_exp <- "sprd_yc_ois3"
# Excepted change in cash rate over next 3 months

## Credit spreads and risk measures
cs_vars <- c("sprd_ir_bab_ois3", "sprd_ir_lbiz_bab3", "sprd_ir_us_cbb_gb10", "us_vix")
cs_vars_main <- c(1,2,3)

## OPTIONAL: Real economic explanatory variables (for backward-looking Taylor Rule)
# "d" for level change, "lnd" for growth rate
sv_names <- c() # "cpii", "gdp", "ur", "ur"
sv_transform <- c() # "lnd", "lnd", "l", "d"
state_vars <- data.frame(sv_names, sv_transform, stringsAsFactors=FALSE)
if (nrow(state_vars)>0) {
  colnames(state_vars) <- c("variable", "transform")
  state_vars <- mutate(state_vars, tf_variable = ifelse(transform=="l", variable,
                                                        paste(transform, variable, sep="_")))
}


### Sample
# Frequency (m or q)
freq <- "q"
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")
smpl_str <- paste(format(as.yearqtr(smpl_start), format = "%Y:Q%q"), "-", 
                 format(as.yearqtr(smpl_end), format = "%Y:Q%q"))
# Sample omission (specify start and end date)
smpl_omit <- c()  # e.g.: as.Date("2008-06-01"), as.Date("2009-06-01")


##### Step 4: Load and clean data and construct missing variables #############
# Load data
load(paste0(data_dir, "data_final.RData"))

# Select variables from dataset
Y <- data_mthly %>% 
  select(Date, fc_mth, !!dep_var, !!lag_dep_var, !!fcast_vars, !!cr_exp, !!cs_vars, 
         !!sv_names)

# Take end-of-PREVIOUS-month financial variables (Board Meeting is at beginning of month)
Y <- Y %>% 
  mutate_at(vars(cr_exp, cs_vars), lag)

# Retain adlp lags of *monthly* dependent variable
ldep_vars <- c()
if (adlp>0){
  for (p in 1:adlp) {
    var_name <- paste0("l", as.character(p), "_", lag_dep_var)
    Y <- mutate(Y, !!var_name := lag(!!as.name(lag_dep_var), n = p))
    ldep_vars <- c(ldep_vars, var_name)
  }
}
rm(var_name)

# Keep forecast months only
if (freq=="q") {
  Y <- filter(Y, fc_mth==1)
}

### Transform state variables
if (nrow(state_vars)>0) {
  Y <- transform_df(Y, state_vars)
}

# Keep only selected sample
Y <- filter(Y, Date>=smpl_start & Date<=smpl_end)
if (length(smpl_omit)>1) {
  Y <- filter(Y, Date>=smpl_omit[1] & Date<=smpl_omit[2])
}


##### Step 5: RR regressions ##################################################
### Baseline models without credit spreads
# Forward-looking Taylor Rule
rr_fw <- rr_regress(Y, dep_var, c(ldep_vars, fcast_vars))

# Forward-looking Taylor Rule controlling for cash rate expectations
rr_fw_exp <- rr_regress(Y, dep_var, c(ldep_vars, fcast_vars, cr_exp))

### Models with credit spreads
rr_fw_csj <- rr_regress(Y, dep_var, c(ldep_vars, fcast_vars, cs_vars[cs_vars_main]))

# Forward-looking Taylor Rule controlling for cash rate expectations
rr_fw_csj_exp <- rr_regress(Y, dep_var, c(ldep_vars, fcast_vars, cs_vars[cs_vars_main], cr_exp))


### Collect results
rr_list <- ls(pattern = "rr_fw")

# Collect fitted values and shocks series (residuals), and estimated models
rr_regs <- list()
for (j in 1:length(rr_list)) {
  assign(rr_list[j], get(eval(rr_list[j]))$rr_est)
  rr_regs[[j]] <- get(eval(rr_list[j]))
}


##### Step 6: Regression results ##############################################
### Regression Results for models including cash rate expectations
# Name columns
collab <- c("BT", "BT-EXP", "BT-CS", "BTCS-EXP")
# Table 2: Estimated Policy Rules with Credit Spreads and Expected Cash Rate Changes
stargazer(rr_regs[c(1,4,2,3)], title = "Estimated Policy Rules with Credit Spreads", 
          align = TRUE, type = "text", no.space = TRUE, dep_var.caption = "", 
          column.labels = collab, model.numbers = FALSE, omit.stat = c("ser", "f"), 
          digits = 3, dep_var.labels = smpl_str, intercept.bottom = FALSE, intercept.top = TRUE)
    # F-test if credit spreads are jointly significant predictors
print(anova(get(eval(rr_list[1])), get(eval(rr_list[4]))))
print(anova(get(eval(rr_list[2])), get(eval(rr_list[3]))))
