##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Collect RBA cash rate changes
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file reads in RBA policy meeting dates and cash rate data
###############################################################################


##### Step 1: Load Forecast Data ##############################################
### Get data from website
data_src <- read_html("https://www.rba.gov.au/statistics/cash-rate/")
cr <- data_src %>% 
  html_node('#datatable') %>% 
  html_table(fill = T) %>% 
  as_tibble()

### Select columns and rename
cr <- cr %>%
  select(-(`Related Documents`))
colnames(cr) <- c("Date", "cr_chg", "cr")

### Keep Data since 1990 only (also removes legend)
cr <- cr %>%
  mutate(year = suppressWarnings(as.numeric(str_sub(Date, -4)))) %>%
  filter(year>=1990) %>%
  select(-(year))

### Convert to Date and numeric and sort by date
cr <- cr %>%
  mutate(cr1 = str_sub(cr, end = 5), cr2 = str_sub(cr, start = -5)) %>% 
  select(-c(cr_chg, cr)) %>% 
  mutate_at(vars(starts_with("cr")), list(~ as.numeric(.))) %>%
  mutate(Date = as.Date(lubridate::parse_date_time(Date, orders = "!d !b !Y"))) %>%
  arrange(Date) %>% 
  mutate(cr = (cr1+cr2)/2) %>% 
  mutate(cr_chg = cr-lag(cr)) %>% 
  select(Date, cr, cr_chg)

### Cash rate before the meeting
cr <- mutate(cr, l.cr = cr - cr_chg)

##### Step 2: Clean Data ######################################################
### Months with double meetings (find second meeting)
cr <- cr %>%
  mutate(MonthYear = floor_date(Date, "month")) %>%
  mutate(cr, M2 = (MonthYear==lag(MonthYear)))

### Move second meeting to next month if no meeting in next month
# Define 'early meetings'
cr <- cr %>%
  mutate(NextMeet = lead(MonthYear)) %>%
  mutate(EarlyM = ((M2==T) & (NextMeet!=(MonthYear %m+% months(1)))))
  
# Change date (EFFECTIVE DATE!!!) of early meetings to first WEDNESDAY of next month
cr <- cr %>%
  mutate(NextMth = floor_date(Date %m+% months(1), "month")) %>%
  mutate(DoW = wday(NextMth)) %>%   # Find with which day of the week the next month starts
  mutate(Date = if_else(EarlyM==T, if_else(DoW<=4, NextMth + days(4-DoW), 
                                           NextMth + days(11-DoW)), Date))

# Drop second meetings when there is a meeting the next month and re-assign changes
cr <- cr %>% 
  filter(cr, !(!is.na(M2) & M2==T & EarlyM==F)) %>%
  mutate(cr, cr_chg = if_else(!is.na(lag(cr)), cr-lag(cr), cr_chg)) %>% 
  mutate(cr, l.cr = cr-cr_chg)

# Change Date from Effective Date to Date of Meeting
cr <- mutate(cr, Date = Date-days(1))

### Keep variables of interest only
cr <- select(cr, Date:l.cr)

##### Step 3: Save data #######################################################
save(cr, file = file.path(data_dir, "cr.Rdata"))
