##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Romer and Romer Monetary Policy Shocks for Australia:
#       The Role of Credit Spreads
# Creator: Ben Beckers
# Date: 22 January 2020
# Description: This file estimates Romer and Romer (2004, RR) monetary policy 
#              shocks for Australia in the following order:
#              1. Purge cash rate changes of the Bank's forecasts, and a set of
#                 variables (real economic, financial) specified by the user.
#              2. Residuals from these regressions serve as monetary policy
#                 shocks.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Table 1: Regression results from Romer and Romer Regression
#          2. Figure 6: Plot of original Bishop & Tulip (2017) RR shock and new
#             credit spread-augmented RR shock
#          3. R- and XLSX-files with RR-type policy shocks
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Source functions
source("Code/functions/funcs.R")
source("Code/functions/rr_regress.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer",
                          "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
dep_var <- "cr_chg"       # Dependent LHS variable (change in cash rate)
lag_dep_var <- "cr"       # Lagged dependent variable (from cr, cr_chg)
adlp <- 1                 # Number of lags of lagged dependent variable

## RBA forecasts and revisions
fcast_vars <- c("fccpiiq2", "fcgdpq2", "fcurq0", "fcrcpiiq2", "fcrgdpq2")
# "fc" for forecast, "fcr" for forecast revision, "q'h'" for quarterly horizon
# "cpii" for underlying inflation, "gdp" for GDP, "ur" for level of unemployment rate, 
# "dur" for change in unemployment rate

## Cash rate expectations (i.e. yield curve)
cr_exp <- "sprd_yc_ois3"

## Credit spreads and risk measures
cs_vars <- c("sprd_ir_bab_ois3", "sprd_ir_lbiz_bab3", "sprd_ir_us_cbb_gb10", "us_vix")
cs_vars_main <- c(1,2,3)

## OPTIONAL: Real economic explanatory variables (for backward-looking Taylor Rule)
# "d" for level change, "lnd" for growth rate
sv_names <- c() # "cpii", "gdp", "ur", "ur", "us_ffr", "rtwi"
sv_transform <- c() # "lnd", "lnd", "l", "d", "l", "l"
state_vars <- data.frame(sv_names, sv_transform, stringsAsFactors=FALSE)
if (nrow(state_vars)>0) {
  colnames(state_vars) <- c("variable", "transform")
  state_vars <- mutate(state_vars, tf_variable = ifelse(transform=="l", variable,
                                                        paste(transform, variable, sep="_")))
}


### Sample
# Frequency (m or q)
freq <- "q"
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")
smpl_str <- paste(format(as.yearqtr(smpl_start), format = "%Y:Q%q"), "-", 
                 format(as.yearqtr(smpl_end), format = "%Y:Q%q"))


##### Step 4: Load and clean data and construct missing variables #############
# Load data
load(paste0(data_dir, "data_final.RData"))

# Select variables from dataset
Y <- data_mthly %>% 
  select(Date, fc_mth, !!dep_var, !!lag_dep_var, !!fcast_vars, !!cr_exp, !!cs_vars, 
         !!sv_names)

# Take end-of-PREVIOUS-month value of financial variables (Board Meeting is at beginning of month)
Y <- Y %>% 
  mutate_at(vars(cr_exp, cs_vars, sv_names), lag)

# Retain adlp lags of *monthly* dependent variable
ldep_vars <- c()
if (adlp>0){
  for (p in 1:adlp) {
    var_name <- paste0("l", as.character(p), "_", lag_dep_var)
    Y <- mutate(Y, !!var_name := lag(!!as.name(lag_dep_var), n = p))
    ldep_vars <- c(ldep_vars, var_name)
  }
}
rm(var_name)

# Keep forecast months only
if (freq=="q") {
  Y <- filter(Y, fc_mth==1)
}

### Transform state variables
if (nrow(state_vars)>0) {
  Y <- transform_df(Y, state_vars)
}

# Keep only selected sample
Y <- filter(Y, Date>=smpl_start & Date<=smpl_end)


##### Step 5: RR regressions ##################################################
### Baseline models without credit spreads
# Forward-looking Taylor Rule
rr_fw <- rr_regress(Y, dep_var, c(ldep_vars, fcast_vars, sv_names))

### Models with credit spreads
# Individual
for (j in 1:length(cs_vars)) {
  mod_name <- paste0("rr_fw_cs", j)
  assign(mod_name, rr_regress(Y, dep_var, c(ldep_vars, fcast_vars, sv_names, cs_vars[j])))
}
rr_fw_csj <- rr_regress(Y, dep_var, c(ldep_vars, fcast_vars, sv_names, cs_vars[cs_vars_main]))


### Collect results
rr_list <- ls(pattern = "rr_ar|rr_fw")

# Collect fitted values and shocks series (residuals), and estimated models
y_fit <- select(Y, cr_chg) %>% 
  zoo(Y$Date)
rr_shocks <- select(Y, cr_chg) %>% 
  zoo(Y$Date)
rr_regs <- list()
for (j in 1:length(rr_list)) {
  y_fit <- cbind(y_fit, get(eval(rr_list[j]))$y_fit)
  rr_shocks <- cbind(rr_shocks, get(eval(rr_list[j]))$rr_shocks)
  colnames(y_fit)[ncol(y_fit)] <- rr_list[j]
  colnames(rr_shocks)[ncol(rr_shocks)] <- rr_list[j]
  assign(rr_list[j], get(eval(rr_list[j]))$rr_est)
  rr_regs[[j]] <- get(eval(rr_list[j]))
}


##### Step 6: First-stage regression results ##################################
### Regression Results for models excluding cash rate 
# Name columns
collab <- "(BT)" 
for (j in 1:length(cs_vars)){
  collab <- c(collab, paste0("(BT-CS", j, ")"))
}
collab <- c(collab, "(BT-CS)")
# Table 1: Estimated Policy Rules with Credit Spreads
stargazer(rr_regs, title = "Estimated Policy Rules with Credit Spreads", 
          align = TRUE, type = "text", no.space = TRUE, dep_var.caption = "", 
          column.labels = collab, model.numbers = FALSE, omit.stat = c("ser", "f"), 
          digits = 2, dep_var.labels = smpl_str, intercept.bottom = FALSE, intercept.top = TRUE)
# F-test if credit spreads are jointly significant predictors
for (j in 2:(length(cs_vars)+2)){
  print(anova(get(eval(rr_list[1])), get(eval(rr_list[j]))))
}


### Figure 6: Cash Rate Changes and Monetary Policy Shocks
rr_shocks <- as_tibble(cbind(Y$Date, as.data.frame(rr_shocks)))
colnames(rr_shocks)[1] <- "Date"
data_fig6 <- select(rr_shocks, Date, cr_chg, rr_fw, rr_fw_csj)
dl_fig6 <- data_fig6 %>%
  pivot_longer(-Date, names_to = "variable", values_to = "value")
if ("arphit" %in% installed.packages()[,"Package"]) {
  fig6 <- arphitgg(aes = agg_aes(x = Date, y = value, group = variable)) + 
    agg_line(data = dl_fig6, colour = c(RBA["Grey10"], RBA["Blue7"], RBA["Orange3"]), 
             lty = c(1,2,2)) +
    agg_units("ppt", panel = "1") +
    agg_ylim(-1, 1, 5, panel = "1") +
    agg_xlim(year(smpl_start), year(smpl_end)) +
    agg_title("Cash Rate Changes and Monetary Policy Shocks") +
    agg_subtitle(paste0("Quarterly, ", smpl_str)) +
    agg_source("Author's calculations") +
    agg_source("Federal Reserve Bank of St. Louis") +
    agg_source("RBA") + 
    agg_label("Actual", x = 1996.5, y = 0.8, colour = RBA["Grey10"], 
              panel = "1") +
    agg_label("Original (BT)*", x = 2005.5, y = -0.4, colour = RBA["Blue7"], 
              panel = "1") +
    agg_label("Augmented (BT-CS)*", x = 2011, y = 0.45, colour = RBA["Orange3"], 
              panel = "1") +
    agg_footnote("Monetary policy shocks (dashed lines) are residuals from regressions BT and BT-CS in Table 1.")
  agg_draw(fig6, filename = paste0(results_dir, "Fig6_Shocks.png")) # 
}
dl_fig6 <- dl_fig6 %>% 
  mutate(label = case_when(variable=="cr_chg" ~ "Actual", 
                           variable=="rr_fw" ~ "Original (BT)", 
                           TRUE ~ "Augmented (BT-CS)"))
data_fig6 <- select(dl_fig6, -c(variable)) %>% 
  pivot_wider(id_cols = Date, names_from = label, values_from = value)
write_xlsx(data_fig6, path = paste(results_dir, "Fig6_Shocks.xlsx", sep=""), 
           col_names = TRUE)


### Step 7: Save output #######################################################
save(list = c("Y", "rr_shocks"), file = paste0(results_dir, "rr_shocks.RData"))
colnames(rr_shocks) <- c("Date", "Cash Rate Change", "Bishop-Tulip (BT)", 
                         "BT-CS1", "BT-CS2", "BT-CS3", "BT-CS4", "BT-CS")
write_xlsx(rr_shocks, path = paste(results_dir, "rr_shocks.xlsx", sep=""), col_names = TRUE)
