##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Romer and Romer Monetary Policy Shocks for Australia:
#       The Role of Credit Spreads
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file estimates alternative Romer and Romer (2004, RR)-type 
#              regression models for Australia in the following order:
#              1. 2-months ahead forecast of the change in the unemployment 
#                 rate, and the lagged change in the cash rate.
#              2. US Fed funds rate, real TWI, and commodity prices as 
#                 additional predictors
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Table D1: Regression results
#          2. R- and XLSX-files with RR-type policy shocks
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Source functions
source("Code/functions/funcs.R")
source("Code/functions/rr_regress.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer",
                          "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
dep_var <- "cr_chg"       # Dependent LHS variable (change in cash rate)
lag_dep_var <- "cr_chg"   # Lagged dependent variable (from cr, cr_chg)
adlp <- 1                 # Number of lags of lagged dependent variable

## RBA forecasts and revisions
fcast_vars <- c("fccpiiq2", "fcgdpq2", "fcdurq2", "fcrcpiiq2", "fcrgdpq2")
# "fc" for forecast, "fcr" for forecast revision, "q'h'" for quarterly horizon
# "inf" for underlying inflation, "gdp" for GDP, "ur" for level of unemployment rate, 
# "dur" for change in unemployment rate

## Cash rate expectations (i.e. yield curve)
cr_exp <- "sprd_yc_ois3"
# Excepted change in cash rate over next 3 months

## Credit spreads and risk measures
cs_vars <- c("sprd_ir_bab_ois3", "sprd_ir_lbiz_bab3", "sprd_ir_us_cbb_gb10", "us_vix")
cs_vars_main <- c(1,2,3)

## Real economic explanatory variables (for backward-looking Taylor Rule)
# "d" for level change, "lnd" for growth rate
sv_names <- c("us_ffr", "ntwi", "commpi")
sv_transform <- c("l", "lnd", "lnd")
state_vars <- data.frame(sv_names, sv_transform, stringsAsFactors=FALSE)
if (nrow(state_vars)>0) {
  colnames(state_vars) <- c("variable", "transform")
  state_vars <- mutate(state_vars, tf_variable = ifelse(transform=="l", variable,
                                                        paste(transform, variable, sep="_")))
}


### Sample
# Frequency (m or q)
freq <- "q"
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")
smpl_str <- paste(format(as.yearqtr(smpl_start), format = "%Y:Q%q"), "-", 
                 format(as.yearqtr(smpl_end), format = "%Y:Q%q"))
# Sample omission (specify start and end date)
smpl_omit <- c()  # e.g.: as.Date("2008-06-01"), as.Date("2009-06-01")


##### Step 4: Load and clean data and construct missing variables #############
# Load data
load(paste0(data_dir, "data_final.RData"))

# Select variables from dataset
Y <- data_mthly %>% 
  select(Date, fc_mth, !!dep_var, !!lag_dep_var, !!fcast_vars, !!cr_exp, !!cs_vars, 
         !!sv_names)

# Take end-of-PREVIOUS-month financial variables (Board Meeting is at beginning of month)
Y <- Y %>% 
  mutate_at(vars(cr_exp, cs_vars), lag)

# Retain adlp lags of *monthly* dependent variable
ldep_vars <- c()
if (adlp>0){
  for (p in 1:adlp) {
    var_name <- paste0("l", as.character(p), "_", lag_dep_var)
    Y <- mutate(Y, !!var_name := lag(!!as.name(lag_dep_var), n = p))
    ldep_vars <- c(ldep_vars, var_name)
  }
}
rm(var_name)

# Keep forecast months only
if (freq=="q") {
  Y <- filter(Y, fc_mth==1)
}

### Transform state variables
if (nrow(state_vars)>0) {
  Y <- transform_df(Y, state_vars)
}

# Keep only selected sample
Y <- filter(Y, Date>=smpl_start & Date<=smpl_end)
if (length(smpl_omit)>1) {
  Y <- filter(Y, Date>=smpl_omit[1] & Date<=smpl_omit[2])
}


##### Step 5: RR regressions ##################################################
### Baseline models without credit spreads
# Forward-looking Taylor Rule
rr_fw <- rr_regress(Y, dep_var, c(ldep_vars, fcast_vars))

### Models with credit spreads
# Individual
for (j in 1:length(cs_vars)) {
  mod_name <- paste0("rr_fw_cs", j)
  assign(mod_name, rr_regress(Y, dep_var, c(ldep_vars, fcast_vars, cs_vars[j])))
}
rr_fw_csj <- rr_regress(Y, dep_var, c(ldep_vars, fcast_vars, cs_vars[cs_vars_main]))

### Models with international variables
rr_fw_aug <- rr_regress(Y, dep_var, c(ldep_vars, fcast_vars, state_vars$tf_variable))
rr_fw_csj_aug <- rr_regress(Y, dep_var, c(ldep_vars, fcast_vars, state_vars$tf_variable, cs_vars[cs_vars_main]))


### Collect results
rr_list <- ls(pattern = "rr_ar|rr_fw")

# Collect fitted values and shocks series (residuals), and estimated models
y_fit <- select(Y, cr_chg) %>% 
  zoo(Y$Date)
rr_shocks_altbase <- select(Y, cr_chg) %>% 
  zoo(Y$Date)
rr_regs <- list()
for (j in 1:length(rr_list)) {
  y_fit <- cbind(y_fit, get(eval(rr_list[j]))$y_fit)
  rr_shocks_altbase <- cbind(rr_shocks_altbase, get(eval(rr_list[j]))$rr_shocks)
  colnames(y_fit)[ncol(y_fit)] <- rr_list[j]
  colnames(rr_shocks_altbase)[ncol(rr_shocks_altbase)] <- rr_list[j]
  assign(rr_list[j], get(eval(rr_list[j]))$rr_est)
  rr_regs[[j]] <- get(eval(rr_list[j]))
}


##### Step 6: First-stage regression results ##################################
### Regression Results for models excluding cash rate 
# Name columns
collab <- c("BT", "BT-Int")
for (j in 1:length(cs_vars)){
  collab <- c(collab, paste0("(BT-CS", j, ")"))
}
collab <- c(collab, "(BT-CS)", "(BT-CS-Int)")
# Table 1: Estimated Policy Rules with Credit Spreads
stargazer(rr_regs, title = "Estimated Policy Rules with Credit Spreads", 
          align = TRUE, type = "text", no.space = TRUE, dep_var.caption = "", report = "vc*",
          column.labels = collab, model.numbers = FALSE, omit.stat = c("ser", "f"), 
          digits = 3, dep_var.labels = smpl_str, intercept.bottom = FALSE, intercept.top = TRUE)
# F-test if credit spreads are jointly significant predictors
for (j in 3:(length(cs_vars)+4)){
  print(anova(get(eval(rr_list[1])), get(eval(rr_list[j]))))
}
rr_shocks_altbase <- as_tibble(cbind(Y$Date, as.data.frame(rr_shocks_altbase)))
colnames(rr_shocks_altbase)[1] <- "Date"
colnames(rr_shocks_altbase)[2:ncol(rr_shocks_altbase)] <- paste0(colnames(rr_shocks_altbase)[2:ncol(rr_shocks_altbase)], "_altbase")


##### Step 7: Remove anticipation effects #####################################
Y2 <- left_join(Y, rr_shocks_altbase, by="Date")
y.zoo <- select(Y2, -Date)
y.zoo <- zoo(y.zoo, Y$Date)

## Second stage regression
shocks_main <- c("rr_fw_csj_altbase", "rr_fw_csj_aug_altbase")
rr_regress_ant <- list()
for (j in 1:length(shocks_main)) {
  # Second-stage regression of RR shocks on cash rate expectations
  eq_rr_ant <- paste0(shocks_main[j], " ~ L(", cr_exp, ", 0)")
  rr_regress_ant[[j]] <- dynlm(as.formula(eq_rr_ant), y.zoo)
  if (j==1){
    rr_shocks_unant <- as.data.frame(rr_regress_ant[[j]]$residuals)
  } else {
    rr_shocks_unant <- cbind(rr_shocks_unant, as.data.frame(rr_regress_ant[[j]]$residuals))
  }
}
## Add to shock series
rr_shocks_unant <- rownames_to_column(rr_shocks_unant, "Date")
colnames(rr_shocks_unant)[2:ncol(rr_shocks_unant)] <- paste0(shocks_main, "_unant")
rr_shocks_unant$Date <- as.Date(rr_shocks_unant$Date)
rr_shocks_altbase <- left_join(rr_shocks_altbase, rr_shocks_unant, by="Date")


##### Step 8: Save output #####################################################
save(list = c("Y", "rr_shocks_altbase"), file = paste0(results_dir, "rr_shocks_altbase.RData"))
write_xlsx(rr_shocks_altbase, path = paste(results_dir, "rr_shocks_altbase.xlsx", sep=""), col_names = TRUE)
