##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Monetary Policy Effects from SVAR: Cash rate vs Bishop and Tulip (2017)
#       policy shocks
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file estimates 2 SVARs to obtain the effect of a cash rate
#              change on inflation to create Figure 1 of the paper.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Figure 1 (for RBA users only, PNG-format)
#          2. Data for Figure 1 in Excel-file.
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Load functions
source("Code/functions/funcs.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer", "plyr",
                          'vars', "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
spec_seed <- 2304
horizon <- 20
cumirf <- FALSE
alpha <- 0.90                 # Confidence interval
varp <- 4                     # VAR lag length 

## Endogenous Variables and Policy Shocks
endo_names <- c("cpii", "gdp", "ur")
endo_transform <- c("ln", "ln", "l")
endo_vars <- data.frame(endo_names, endo_transform, stringsAsFactors=FALSE)
colnames(endo_vars) <- c("variable", "transform")
endo_vars <- mutate(endo_vars, tf_variable = ifelse(transform=="l", variable,
                                                    paste(transform, variable, sep="_")))

shocks <- c("rr_fw")
# Order of variables (1 or length(endo_names)+1)
pos_shock <- length(endo_names)+1

### Sample
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")
# Sample omission (specify start and end date)
smpl_omit <- c()  # e.g.: as.Date("2008-06-01"), as.Date("2009-06-01")


##### Step 4: Load and clean data and construct missing variables #############
### Policy Shocks
load(paste0(results_dir, "rr_shocks.RData"))
# Select shocks
rr_shocks <- select(rr_shocks, Date, !!shocks)
# Cumulate shocks
rr_shocks <- mutate_at(rr_shocks, vars(-Date), cumsum)
shocks <- c("cr", shocks)


### VAR data
load(paste0(data_dir, "data_final.RData"))

# Select variables from dataset
Y <- data_mthly %>% 
  select(Date, cr, !!endo_names, fc_mth)

# Keep forecast months only
Y <- filter(Y, fc_mth==1) %>% 
  select(-fc_mth)

# Transform endogenous variables
Y <- transform_df(Y, endo_vars)

# Keep only selected sample
Y <- filter(Y, Date>=smpl_start & Date<=smpl_end)
if (length(smpl_omit)>1) {
  Y <- filter(Y, Date>=smpl_omit[1] & Date<=smpl_omit[2])
}

### Merge shocks and VAR data
Y <- left_join(Y, rr_shocks, by="Date")


##### Step 5: SVAR Regressions ################################################
for (j in 1:length(shocks)) {
  # Select variables
  mp_shk <- shocks[j]
  y <- select(Y, !!endo_vars$tf_variable, !!mp_shk)
  y[is.na(y), mp_shk] <- 0
  
  # Estimate VAR
  var_est <- VAR(y, varp, type = "const") # , exogen = x
  
  # Obtain IRF
  irf_est <- irf(var_est, impulse = mp_shk, cumulative = cumirf, boot = TRUE,
                 n.ahead = horizon, ortho = TRUE, ci = alpha, runs = 5000, seed = spec_seed)
  
  # Standardise and bind together
  IRFs <- as.data.frame(irf_est$irf[[1]]/irf_est$irf[[1]][1,mp_shk]) %>% 
    as_tibble() %>% 
    mutate(horizon = seq(0,horizon))
  IRFs_lwr <- as.data.frame(irf_est$Lower[[1]]/irf_est$Lower[[1]][1,mp_shk]) %>% 
    as_tibble() %>% 
    rename_all(function(x) paste0(x,"_lwr")) %>% 
    mutate(horizon = seq(0,horizon))
  IRFs_upr <- as.data.frame(irf_est$Upper[[1]]/irf_est$Upper[[1]][1,mp_shk]) %>% 
    as_tibble() %>% 
    rename_all(function(x) paste0(x,"_upr")) %>% 
    mutate(horizon = seq(0,horizon))
  IRFs <- join_all(list(IRFs, IRFs_lwr, IRFs_upr), by = "horizon", type = "full")
  # Obtain percentage point/basis point changes
  IRFs <- mutate_at(IRFs, vars(contains("ln")), function(x) {100*x}) %>% 
    mutate_at(vars(contains(mp_shk)), function(x) {100*x})
  
  # Transform to long format and re-order columns by variable names
  IRFs_lg <- pivot_longer(IRFs, -horizon, names_to = "variable", values_to = "value") %>% 
    arrange(variable, horizon) %>% 
    mutate(shock = mp_shk)
  
  if (j==1) {
    IRFs_all_lg <- IRFs_lg
  } else {
    IRFs_all_lg <- bind_rows(IRFs_all_lg, IRFs_lg)
  }
}


##### Step 6: Plot IRFs #######################################################
if ("arphit" %in% installed.packages()[,"Package"]) {
  fig1 <- arphitgg(aes = agg_aes(x = horizon, y = value, group = variable), layout = "2v"
                   , dropxlabel = TRUE, showallxlabels = FALSE) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'ln_cpii') & shock == "cr"), panel = "1",
             colour = c(RBA["Aqua8"], RBA["Red6"], RBA["Red6"]), lty = c(1,2,2), lwd = rep(3,3)) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'ln_cpii') & shock == "rr_fw"), panel = "2",
             colour = c(RBA["Aqua8"], RBA["Red6"], RBA["Red6"]), lty = c(1,2,2), lwd = rep(3,3)) + 
    agg_title("Standard SVAR**", panel = "1") +
    agg_title("Bishop and Tulip Shock***", panel = "2") +
    agg_ylim(-1, 2, 7) +
    agg_title("Price Level Response to Contractionary Monetary Policy Shock") +
    agg_subtitle("CPI (Underlying), 100 basis points cash rate shock, cumulative quarterly response*") +
    agg_source("ABS") +
    agg_source("Author's calculations") +
    agg_source("RBA") + 
    agg_footnote("Impulse responses from a recursive SVAR(4) including log underlying CPI, log real GDP, the unemployment rate, and a monetary policy variable; dashed lines show 90% confidence bands (bootstrapped); 1994:Q1-2018:Q4.") + 
    agg_footnote("Policy variable is the cash rate") + 
    agg_footnote("Policy variable is the cumulated policy shock from Bishop and Tulip (2017, updated)")
  agg_draw(fig1, filename = paste0(results_dir, "Fig1_SVAR_Intro.png"))
}

### Export graph data
IRFs_all <- filter(IRFs_all_lg, str_detect(variable, 'ln_cpii')) %>% 
  mutate(label = case_when((variable=='ln_cpii' & shock == 'cr') ~ "Point (Cash Rate, left panel)", 
                           (variable=='ln_cpii_lwr' & shock == 'cr') ~ "Lower (Cash Rate, left panel)", 
                           (variable=='ln_cpii_upr' & shock == 'cr') ~ "Upper (Cash Rate, left panel)", 
                           (variable=='ln_cpii' & shock == 'rr_fw') ~ "Point (BT Shock, right panel)", 
                           (variable=='ln_cpii_lwr' & shock == 'rr_fw') ~ "Lower (BT Shock, right panel)", 
                           TRUE ~ "Upper (BT Shock, right panel)")) %>% 
  select(-c(variable, shock)) %>% 
  pivot_wider(names_from = label, values_from = value)
write_xlsx(IRFs_all, path = paste(results_dir, "Fig1_SVAR_Intro.xlsx", sep=""), col_names = TRUE)
