##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Monetary Policy Effects from SVARs: The Role of Financial Conditions in
#       the reduced-form VAR
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file estimates the SVARs to obtain the results to create 
#              Figure 10 of the paper.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Figure 10 (for RBA users only, PNG-format)
#          2. Data for Figure 10 in Excel-file.
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Load functions
source("Code/functions/funcs.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer", "plyr",
                          'vars', "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
spec_seed <- 2304
horizon <- 20
cumirf <- FALSE
alpha <- 0.90               # Confidence interval
varp <- 4

## Endogenous Variables and Policy Shocks
endo_names <- c("cpii", "gdp", "ur", "sprd_ir_lbiz_bab3", "sprd_ir_bab_ois3")
endo_transform <- c("ln", "ln", "l", "l", "l")
endo_vars_set <- data.frame(endo_names, endo_transform, stringsAsFactors=FALSE)
colnames(endo_vars_set) <- c("variable", "transform")
endo_vars_set <- mutate(endo_vars_set, tf_variable = ifelse(transform=="l", variable,
                                                        paste(transform, variable, sep="_")))

shocks <- c("rr_fw", "rr_fw_csj_unant")
# Order of variables (1 or length(endo_names)+1)
pos_shock <- length(endo_names)+1

### Sample
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")
# Sample omission (specify start and end date)
smpl_omit <- c()  # e.g.: as.Date("2008-06-01"), as.Date("2009-06-01")


##### Step 4: Load and clean data and construct missing variables #############
### Policy Shocks
load(paste0(results_dir, "rr_shocks.RData"))
if (any(str_detect(shocks, "unant"))) {
  load(paste0(results_dir, "rr_shocks_unant.RData"))
  rr_shocks <- left_join(rr_shocks, rr_shocks_unant, by="Date")
}
# Select shocks
rr_shocks <- select(rr_shocks, Date, !!shocks)
# Cumulate shocks
rr_shocks <- mutate_at(rr_shocks, vars(-Date), cumsum)


### VAR data
load(paste0(data_dir, "data_final.RData"))

# Select variables from dataset
Y <- data_mthly %>% 
  select(Date, !!endo_names, fc_mth)

# Keep forecast months only
Y <- filter(Y, fc_mth==1) %>% 
  select(-fc_mth)

# Transform endogenous variables
Y <- transform_df(Y, endo_vars_set)

# Keep only selected sample
Y <- filter(Y, Date>=smpl_start & Date<=smpl_end)
if (length(smpl_omit)>1) {
  Y <- filter(Y, Date>=smpl_omit[1] & Date<=smpl_omit[2])
}

### Merge shocks and VAR data
Y <- left_join(Y, rr_shocks, by="Date")


##### Step 5: SVAR Regressions ################################################
for (v in 1:2) {
  for (j in 1:length(shocks)) {
    # Select variables
    mp_shk <- shocks[j]
    if (v==1) {
      endo_vars <- filter(endo_vars_set, variable %in% c("cpii", "ur", "gdp"))
      y <- select(Y, !!endo_vars$tf_variable, !!mp_shk)
    } else if (v==2) {
      endo_vars <- endo_vars_set
      y <- select(Y, !!endo_vars$tf_variable[1:3], !!mp_shk, !!endo_vars$tf_variable[4:5])
    }
  
    y[is.na(y), mp_shk] <- 0
    
    # Estimate VAR
    var_est <- VAR(y, varp, type = "const") # , exogen = x
    
    # Obtain IRF
    irf_est <- irf(var_est, impulse = mp_shk, cumulative = cumirf, boot = TRUE,
                   n.ahead = horizon, ortho = TRUE, ci = alpha, runs = 5000, seed = spec_seed)
    
    # Standardise and bind together
    IRFs <- as.data.frame(irf_est$irf[[1]]/irf_est$irf[[1]][1,mp_shk]) %>% 
      as_tibble() %>% 
      mutate(horizon = seq(0,horizon))
    if (v==1) {
      IRFs_lwr <- as.data.frame(irf_est$Lower[[1]]/irf_est$Lower[[1]][1,mp_shk]) %>% 
        as_tibble() %>% 
        rename_all(function(x) paste0(x,"_lwr")) %>% 
        mutate(horizon = seq(0,horizon))
      IRFs_upr <- as.data.frame(irf_est$Upper[[1]]/irf_est$Upper[[1]][1,mp_shk]) %>% 
        as_tibble() %>% 
        rename_all(function(x) paste0(x,"_upr")) %>% 
        mutate(horizon = seq(0,horizon))
      IRFs <- join_all(list(IRFs, IRFs_lwr, IRFs_upr), by = "horizon", type = "full")
    }
    # Obtain percentage point/basis point changes
    IRFs <- mutate_at(IRFs, vars(contains("ln")), function(x) {100*x})
    
    # Transform to long format and re-order columns by variable names
    IRFs_lg <- pivot_longer(IRFs, -horizon, names_to = "variable", values_to = "value") %>% 
      arrange(variable, horizon) %>% 
      mutate(shock = mp_shk) %>% 
      mutate(var_spec = paste0("var",v))
    
    # Combine results
    if (v==1 & j==1) {
      IRFs_all_lg <- IRFs_lg
    } else {
      IRFs_all_lg <- bind_rows(IRFs_all_lg, IRFs_lg)
    }
  }
}
IRFs_all <- pivot_wider(IRFs_all_lg, names_from = c(variable, shock, var_spec), 
                        names_sep = "_", values_from = value)
IRF_plot_lg <- IRFs_all %>% 
  select(matches('horizon|cpii|ur')) %>% 
  pivot_longer(-horizon, names_to = "variable", values_to = "value") %>% 
  arrange(variable, horizon)

##### Step 6: Plot IRFs #######################################################
### CPI and UR
if ("arphit" %in% installed.packages()[,"Package"]) {
  fig10 <- arphitgg(aes = agg_aes(x = horizon, y = value, group = variable), layout = "2b2"
                   , dropxlabel = TRUE, showallxlabels = FALSE) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ln_cpii') & str_detect(variable, 'rr_fw_csj', negate=T)), 
             panel = "1", lty = c(2,1,1,2), lwd = rep(3,4),
             colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Aqua5"], RBA["Red6"])) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ln_cpii') & str_detect(variable, 'rr_fw_csj')), 
             panel = "2", lty = c(2,1,1,2), lwd = rep(3,4),
             colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Aqua5"], RBA["Red6"])) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ur') & str_detect(variable, 'rr_fw_csj', negate=T)), 
             panel = "3", lty = c(2,1,1,2), lwd = rep(3,4),
             colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Aqua5"], RBA["Red6"])) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ur') & str_detect(variable, 'rr_fw_csj')), 
             panel = "4", lty = c(2,1,1,2), lwd = rep(3,4),
             colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Aqua5"], RBA["Red6"])) + 
    agg_yaxislabel("CPI", panel = "1") + 
    agg_yaxislabel("Unemployment", panel = "3") + 
    agg_title("BT Shock**", panel = "1") +
    agg_title("Unanticipated BT-CS Shock***", panel = "2") +
    agg_units("ppt", panel = c("3","4")) +
    agg_ylim(-2, 2, 5, panel = c("1","2")) +
    agg_ylim(-1.5, 1.5, 7, panel = c("3","4")) + 
    agg_title("SVAR with augmented by Financial Conditions") +
    agg_subtitle("Cumulative quarterly response to 100 basis points policy shock*") +
    agg_source("ABS") +
    agg_source("Author's calculations") +
    agg_source("RBA") + 
    agg_footnote("Responses from baseline SVAR (darker solid lines and confidence intervals) and SVAR augmented by money market risk spread and large business lending rate spread (lighter solid lines).") + 
    agg_footnote("Original, anticipated policy shock used by Bishop and Tulip (2017, baseline SVAR responses as shown in left panels of Figure 7).") + 
    agg_footnote("New, unanticipated policy shock purged of the response to credit spreads (baseline SVAR responses as shown in right panels of Figure 9).")
  agg_draw(fig10, filename = paste0(results_dir, "Fig10_SVAR_FinCon.png"))
}

### Export graph data
IRFs_SVAR_FinCon <- mutate(IRF_plot_lg, panel = case_when((variable=='ln_cpii_rr_fw_var1') ~ 1, 
                           (variable=='ln_cpii_lwr_rr_fw_var1') ~ 1, 
                           (variable=='ln_cpii_upr_rr_fw_var1') ~ 1, 
                           (variable=='ln_cpii_rr_fw_var2') ~ 1, 
                           (variable=='ln_cpii_rr_fw_csj_unant_var1') ~ 2, 
                           (variable=='ln_cpii_lwr_rr_fw_csj_unant_var1') ~ 2, 
                           (variable=='ln_cpii_upr_rr_fw_csj_unant_var1') ~ 2, 
                           (variable=='ln_cpii_rr_fw_csj_unant_var2') ~ 2,
                           (variable=='ur_rr_fw_var1') ~ 3, 
                           (variable=='ur_lwr_rr_fw_var1') ~ 3, 
                           (variable=='ur_upr_rr_fw_var1') ~ 3, 
                           (variable=='ur_rr_fw_var2') ~ 3, 
                           (variable=='ur_rr_fw_csj_unant_var1') ~ 4, 
                           (variable=='ur_lwr_rr_fw_csj_unant_va1') ~ 4, 
                           (variable=='ur_upr_rr_fw_csj_unant_var1') ~ 4, 
                           TRUE ~ 4)) %>% 
  arrange(panel, variable, horizon) %>% 
  mutate(label = case_when((variable=='ln_cpii_rr_fw_var2') ~ "CPI (Aug-VAR, BT Shock, Pan1)", 
                           (variable=='ln_cpii_lwr_rr_fw_var1') ~ "CPI Lower (Bench-VAR, BT Shock, Pan1)", 
                           (variable=='ln_cpii_upr_rr_fw_var1') ~ "CPI Upper (Bench-VAR, BT Shock, Pan1)", 
                           (variable=='ln_cpii_rr_fw_var1') ~ "CPI (Bench-VAR, BT Shock, Pan1)", 
                           (variable=='ln_cpii_rr_fw_csj_unant_var2') ~ "CPI (Aug-VAR, BT-CS Shock, Pan2)", 
                           (variable=='ln_cpii_lwr_rr_fw_csj_unant_var1') ~ "CPI Lower (Bench-VAR, BT-CS Shock, Pan2)", 
                           (variable=='ln_cpii_upr_rr_fw_csj_unant_var1') ~ "CPI Upper (Bench-VAR, BT-CS Shock, Pan2)", 
                           (variable=='ln_cpii_rr_fw_csj_unant_var1') ~ "CPI (Bench-VAR, BT-CS Shock, Pan2)",
                           (variable=='ur_rr_fw_var2') ~ "UR (Aug-VAR, BT Shock, Pan3)", 
                           (variable=='ur_lwr_rr_fw_var1') ~ "UR Lower (Bench-VAR, BT Shock, Pan3)", 
                           (variable=='ur_upr_rr_fw_var1') ~ "UR Upper (Bench-VAR, BT Shock, Pan3)", 
                           (variable=='ur_rr_fw_var1') ~ "UR (Bench-VAR, BT Shock, Pan3)", 
                           (variable=='ur_rr_fw_csj_unant_var2') ~ "UR (Aug-VAR, BT-CS Shock, Pan4)", 
                           (variable=='ur_lwr_rr_fw_csj_unant_var1') ~ "UR Lower (Bench-VAR, BT-CS Shock, Pan4)", 
                           (variable=='ur_upr_rr_fw_csj_unant_var1') ~ "UR Upper (Bench-VAR, BT-CS Shock, Pan4)", 
                           TRUE ~ "UR (Bench-VAR, BT-CS Shock, Pan4)")) %>% 
  select(-c(variable, panel)) %>% 
  pivot_wider(names_from = label, values_from = value)
write_xlsx(IRFs_SVAR_FinCon, path = paste(results_dir, "Fig10_SVAR_FinCon.xlsx", sep=""), col_names = TRUE)
