##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Monetary Policy Effects from SVARs: Robustness to alternative RR
#       regression specification
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file estimates the SVARs to obtain the results to create 
#              Figure D3 of the paper.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Figure D3 (for RBA users only, PNG-format)
#          2. Data for Figure D3 in Excel-file.
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Load functions
source("Code/functions/funcs.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer", "plyr",
                          'vars', "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
spec_seed <- 2304
horizon <- 20
cumirf <- FALSE
alpha <- 0.90               # Confidence interval
varp <- 4                     # VAR lag length 

## Endogenous Variables and Policy Shocks
endo_names <- c("cpii", "gdp", "ur")
endo_transform <- c("ln", "ln", "l")
endo_vars <- data.frame(endo_names, endo_transform, stringsAsFactors=FALSE)
colnames(endo_vars) <- c("variable", "transform")
endo_vars <- mutate(endo_vars, tf_variable = ifelse(transform=="l", variable,
                                                    paste(transform, variable, sep="_")))

### Sample
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")


##### Step 4: Load and clean data and construct missing variables #############
### Policy Shocks
load(paste0(results_dir, "rr_shocks.RData"))
rr_shocks <- select(rr_shocks, Date, rr_fw)
load(paste0(results_dir, "rr_shocks_unant.RData"))
rr_shocks_unant <- select(rr_shocks_unant, Date, rr_fw_csj_unant)
load(paste0(results_dir, "rr_shocks_altbase.RData"))
rr_shocks_altbase <- select(rr_shocks_altbase, Date, rr_fw_altbase, matches("rr_fw_altbase|rr_fw_aug|_unant"))
rr_shocks <- join_all(list(rr_shocks, rr_shocks_unant, rr_shocks_altbase), by="Date")
shocks <- colnames(rr_shocks[2:ncol(rr_shocks)])

# Cumulate shocks
rr_shocks <- mutate_at(rr_shocks, vars(-Date), cumsum)

### VAR data
load(paste0(data_dir, "data_final.RData"))

# Select variables from dataset
Y <- data_mthly %>% 
  select(Date, !!endo_names, fc_mth)

# Keep forecast months only
Y <- filter(Y, fc_mth==1) %>% 
  select(-fc_mth)

# Transform endogenous variables
Y <- transform_df(Y, endo_vars)

# Keep only selected sample
Y <- filter(Y, Date>=smpl_start & Date<=smpl_end)

### Merge shocks and VAR data
Y <- left_join(Y, rr_shocks, by="Date")

##### Step 5: SVAR Regressions ################################################
for (j in 1:length(shocks)) {
  # Select variables
  mp_shk <- shocks[j]
  y <- select(Y, !!endo_vars$tf_variable, !!mp_shk)
  y[is.na(y), mp_shk] <- 0
  
  # Estimate VAR
  var_est <- VAR(y, varp, type = "const") # , exogen = x
  
  # Obtain IRF
  irf_est <- irf(var_est, impulse = mp_shk, cumulative = cumirf, boot = TRUE,
                 n.ahead = horizon, ortho = TRUE, ci = alpha, runs = 5000, seed = spec_seed)
  
  # Standardise and bind together
  IRFs <- as.data.frame(irf_est$irf[[1]]/irf_est$irf[[1]][1,mp_shk]) %>% 
    as_tibble() %>% 
    mutate(horizon = seq(0,horizon))
  if (mp_shk=="rr_fw" | mp_shk=="rr_fw_csj_unant") {
    IRFs_lwr <- as.data.frame(irf_est$Lower[[1]]/irf_est$Lower[[1]][1,mp_shk]) %>% 
      as_tibble() %>% 
      rename_all(function(x) paste0(x,"_lwr")) %>% 
      mutate(horizon = seq(0,horizon))
    IRFs_upr <- as.data.frame(irf_est$Upper[[1]]/irf_est$Upper[[1]][1,mp_shk]) %>% 
      as_tibble() %>% 
      rename_all(function(x) paste0(x,"_upr")) %>% 
      mutate(horizon = seq(0,horizon))
    IRFs <- join_all(list(IRFs, IRFs_lwr, IRFs_upr), by = "horizon", type = "full")
  }
  # Obtain percentage point/basis point changes
  IRFs <- mutate_at(IRFs, vars(contains("ln")), function(x) {100*x})
  
  # Transform to long format and re-order columns by variable names
  IRFs_lg <- pivot_longer(IRFs, -horizon, names_to = "variable", values_to = "value") %>% 
    arrange(variable, horizon) %>% 
    mutate(shock = mp_shk)
  
  if (j==1) {
    IRFs_all_lg <- IRFs_lg
  } else {
    IRFs_all_lg <- bind_rows(IRFs_all_lg, IRFs_lg)
  }
}
IRFs_all <- pivot_wider(IRFs_all_lg, names_from = c(variable, shock), names_sep = "_", 
                        values_from = value)
IRF_plot_lg <- IRFs_all %>% 
  select(matches('horizon|cpii|ur')) %>% 
  pivot_longer(-horizon, names_to = "variable", values_to = "value") %>% 
  arrange(variable, horizon)


##### Step 6: Plot IRFs #######################################################
### CPI and UR
if ("arphit" %in% installed.packages()[,"Package"]) {
  figD3 <- arphitgg(aes = agg_aes(x = horizon, y = value, group = variable), layout = "2b2"
                   , dropxlabel = TRUE, showallxlabels = FALSE) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ln_cpii') & str_detect(variable, 'rr_fw_csj', negate=T)), 
             panel = "1", lty = c(2,1,1,1,2), lwd = rep(3,5), colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Aqua4"], RBA["Green8"], RBA["Red6"])) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ln_cpii') & str_detect(variable, 'rr_fw_csj')), 
             panel = "2", lty = c(2,1,1,1,2), lwd = rep(3,5), colour = c(RBA["Red6"], RBA["Aqua4"], RBA["Green8"], RBA["Aqua8"], RBA["Red6"])) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ur') & str_detect(variable, 'rr_fw_csj', negate=T)), 
             panel = "3", lty = c(2,1,1,1,2), lwd = rep(3,5), colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Aqua4"], RBA["Green8"], RBA["Red6"])) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ur') & str_detect(variable, 'rr_fw_csj')), 
             panel = "4", lty = c(2,1,1,1,2), lwd = rep(3,5), colour = c(RBA["Red6"], RBA["Aqua4"], RBA["Green8"], RBA["Aqua8"], RBA["Red6"])) + 
    agg_yaxislabel("CPI", panel = "1") + 
    agg_yaxislabel("Unemployment", panel = "3") + 
    agg_title("BT Shock", panel = "1") +
    agg_title("Unanticipated BT-CS Shock", panel = "2") +
    agg_units("ppt", panel = c("3","4")) +
    agg_ylim(-3, 3, 7, panel = c("1","2")) +
    agg_ylim(-2, 1, 4, panel = c("3","4")) +
    agg_rename_series(list("Baseline**" = "ln_cpii_rr_fw", "Alternative Forecasts***" = "ln_cpii_rr_fw_altbase",
                           "International Variables****" = "ln_cpii_rr_fw_aug_altbase", 
                           "<NA>" = "ln_cpii_lwr_rr_fw", "<NA>" = "ln_cpii_upr_rr_fw",
                           "<NA>" = "ln_cpii_rr_fw_csj_unant", "<NA>" = "ln_cpii_rr_fw_csj_altbase_unant",
                           "<NA>" = "ln_cpii_rr_fw_csj_aug_altbase_unant", 
                           "<NA>" = "ln_cpii_lwr_rr_fw_csj_unant", "<NA>" = "ln_cpii_upr_rr_fw_csj_unant",
                           "<NA>" = "ur_rr_fw", "<NA>" = "ur_rr_fw_altbase", "<NA>" = "ur_rr_fw_aug_altbase",
                           "<NA>" = "ur_lwr_rr_fw", "<NA>" = "ur_upr_rr_fw",
                           "<NA>" = "ur_rr_fw_csj_unant", "<NA>" = "ur_rr_fw_csj_altbase_unant",
                           "<NA>" = "ur_rr_fw_csj_aug_altbase_unant", 
                           "<NA>" = "ur_lwr_rr_fw_csj_unant", "<NA>" = "ur_upr_rr_fw_csj_unant")) + 
    agg_legend(x=0.23, y=0.66) + 
    agg_title("SVAR Robustness - Different Taylor-Rule Specifications") +
    agg_subtitle("Cumulative quarterly response to 100 bps policy shock*") +
    agg_source("ABS") +
    agg_source("Author's calculations") +
    agg_source("RBA") + 
    agg_footnote("Impulse responses to policy shocks obtained from alternative baseline Taylor-rule regression (Table D1). Left panel shows responses to original, anticipated Bishop and Tulip (2017) shock. Right panel shows responses to new preferred unanticipated shock. For further notes see Figures 7 and 8.") +
    agg_footnote("Original Taylor-rule specification used by Bishop and Tulip (2017).") +
    agg_footnote("Alternative Taylor-rule specification (BT-CS in Table D1).") +
    agg_footnote("Alternative Taylor-rule specification including international conditions (BT-CS-Int in Table D1).")
  agg_draw(figD3, filename = paste0(results_dir, "FigD3_SVAR_AltBase.png"))
}

### Export graph data
IRFs_altbase <- mutate(IRF_plot_lg, panel = case_when((variable=='ln_cpii_rr_fw') ~ 1, 
                                                      (variable=='ln_cpii_lwr_rr_fw') ~ 1, 
                                                      (variable=='ln_cpii_upr_rr_fw') ~ 1, 
                                                      (variable=='ln_cpii_rr_fw_altbase') ~ 1, 
                                                      (variable=='ln_cpii_rr_fw_aug_altbase') ~ 1,  
                                                      (variable=='ln_cpii_rr_fw_csj_unant') ~ 2, 
                                                      (variable=='ln_cpii_lwr_rr_fw_csj_unant') ~ 2, 
                                                      (variable=='ln_cpii_upr_rr_fw_csj_unant') ~ 2, 
                                                      (variable=='ln_cpii_rr_fw_csj_altbase_unant') ~ 2, 
                                                      (variable=='ln_cpii_rr_fw_csj_aug_altbase_unant') ~ 2, 
                                                      (variable=='ur_rr_fw') ~ 3, 
                                                      (variable=='ur_lwr_rr_fw_aug_altbase') ~ 3, 
                                                      (variable=='ur_upr_rr_fw_aug_altbase') ~ 3, 
                                                      (variable=='ur_rr_fw_altbase') ~ 3, 
                                                      (variable=='ur_rr_fw_aug_altbase') ~ 3, 
                                                      (variable=='ur_rr_fw_csj_unant') ~ 4, 
                                                      (variable=='ur_lwr_rr_fw_csj_unant') ~ 4, 
                                                      (variable=='ur_upr_rr_fw_csj_unant') ~ 4, 
                                                      (variable=='ur_rr_fw_csj_altbase_unant') ~ 4, 
                                                      TRUE ~ 4)) %>% 
  arrange(panel, variable, horizon) %>% 
  mutate(label = case_when((variable=='ln_cpii_rr_fw') ~ "CPI (BT Shock, Benchmark, Pan1)", 
                           (variable=='ln_cpii_lwr_rr_fw') ~ "CPI Lower (BT Shock, Benchmark, Pan1)", 
                           (variable=='ln_cpii_upr_rr_fw') ~ "CPI Upper (BT Shock, Benchmark, Pan1)", 
                           (variable=='ln_cpii_rr_fw_altbase') ~ "CPI (BT Shock, Alt Fcast, Pan1)", 
                           (variable=='ln_cpii_rr_fw_aug_altbase') ~ "CPI (BT Shock, Int Vars, Pan1)", 
                           (variable=='ln_cpii_rr_fw_csj_unant') ~ "CPI (BT-CS Shock, Benchmark, Pan2)", 
                           (variable=='ln_cpii_lwr_rr_fw_csj_unant') ~ "CPI Lower (BT-CS Shock, Benchmark, Pan2)", 
                           (variable=='ln_cpii_upr_rr_fw_csj_unant') ~ "CPI Upper (BT-CS Shock, Benchmark, Pan2)", 
                           (variable=='ln_cpii_rr_fw_csj_altbase_unant') ~ "CPI (BT-CS Shock, Alt Fcast, Pan2)", 
                           (variable=='ln_cpii_rr_fw_csj_aug_altbase_unant') ~ "CPI (BT-CS Shock, Int Vars, Pan2)", 
                           (variable=='ur_rr_fw') ~ "UR (BT Shock, Benchmark, Pan3)", 
                           (variable=='ur_lwr_rr_fw') ~ "UR Lower (BT Shock, Benchmark, Pan3)", 
                           (variable=='ur_upr_rr_fw') ~ "UR Upper (BT Shock, Benchmark, Pan3)", 
                           (variable=='ur_rr_fw_altbase') ~ "UR (BT Shock, Alt Fcast, Pan3)", 
                           (variable=='ur_rr_fw_aug_altbase') ~ "UR (BT Shock, Int Vars, Pan3)", 
                           (variable=='ur_rr_fw_csj_unant') ~ "UR (BT-CS Shock, Benchmark, Pan4)", 
                           (variable=='ur_lwr_rr_fw_csj_unant') ~ "UR Lower (BT-CS Shock, Benchmark, Pan4)", 
                           (variable=='ur_upr_rr_fw_csj_unant') ~ "UR Upper (BT-CS Shock, Benchmark, Pan4)", 
                           (variable=='ur_rr_fw_csj_altbase_unant') ~ "UR (BT-CS Shock, Alt Fcast, Pan4)", 
                           TRUE ~ "UR (BT-CS Shock, Int Vars, Pan4)")) %>% 
  select(-c(variable, panel)) %>% 
  pivot_wider(names_from = label, values_from = value)
write_xlsx(IRFs_altbase, path = paste(results_dir, "FigD3_SVAR_AltBase.xlsx", sep=""), col_names = TRUE)
