### Install Packages
check_and_install_packs <- function(pack_list, force_install = FALSE, silent = FALSE) {
  # Remove duplicates
  packs_to_install <- unique(pack_list)
  # Check if already installed
  if (!force_install) {
    packs_to_install <- packs_to_install[!(packs_to_install %in% installed.packages()[,"Package"])]
  }
  # Install packages
  if (length(packs_to_install) > 0) {
    install_packages(packs_to_install)
  }
  
  # Print status
  load_status <- sapply(pack_list, require, character.only = TRUE)
  if (length(pack_list) > 0 & !silent) {
    print(load_status)
  }
}


### Load Metadata
get_metadata <- function(data_dir) {
  metadata_xlsx <- file.path(data_dir, '_series.xlsx')
  metadata_xlsx_sheets <- readxl::excel_sheets(metadata_xlsx)
  metadata_df_list <- lapply(metadata_xlsx_sheets,
                             function(x) {
                               readxl::read_excel(metadata_xlsx, x)
                             })
  names(metadata_df_list) <- metadata_xlsx_sheets
  
  metadata <- bind_rows(metadata_df_list)
  
  return(metadata)
}


### Rename variables from Series ID to Mnemonic according to Metadata
lookup_rename <- function(df, df_lookup, column_mnemonic, column_series_ID) {
  new_names <- df_lookup %>% 
    select(column_mnemonic, column_series_ID) %>% 
    filter(!!rlang::sym(column_series_ID) %in% names(df)) %>% 
    select(column_mnemonic)
  new_names[[column_mnemonic]]
  
  return(new_names)
}

### Deflate series
deflate_ser <- function(x, deflator) (
  100*x/deflator
)

### Compute differences
diff2df <- function(x, l=1) {
  y <- x-lag(x, n=l)
  return(y)
}

### Compute log-differences
logdiff2df <- function(x, l = 1) {
  y <- log(x)-lag(log(x), n=l)
  return(y)
}

### Transform range of variables according to specified transformation code
transform_df <- function(data, var_tf_list, l=1) {
  y <- data %>%
    pivot_longer(-Date, names_to = "variable", values_to = "value")
  # Load transformation
  y <- left_join(y, var_tf_list, by="variable") %>% 
    select(Date, variable, value, transform) %>%
    arrange(variable, transform, Date) %>% 
    mutate(transform = ifelse(is.na(transform), "l", transform))
  # Apply transformations
  y <- y %>% 
    group_by(variable, transform) %>% 
    mutate(value = case_when(
      transform=="d" ~ diff2df(value, l),
      transform=="ln" ~ log(value),
      transform=="lnd" ~ 100*logdiff2df(value, l),
      transform=="l" ~ value)
    ) %>% 
    ungroup()
  # Bring back to wide format
  Y <- pivot_wider(y, names_from = c(transform, variable), values_from = value, names_sep = "_") %>% 
    rename_at(vars(starts_with("l_")), list(~ str_replace(., "l_", "")))
  for (j in 1:nrow(var_tf_list)) {
    trash <- which(str_detect(colnames(Y), var_tf_list$variable[j]))
    colnames(Y)[trash] <- var_tf_list$tf_variable[j]
  }
  return(Y)
}

### Add NA to middle of dataset
fill_na <- function(x, df_dest) {
  first_na <- min(which(is.na(df_dest)))
  last_na <- max(which(is.na(df_dest)))
  y <- as.data.frame(rbind(as.matrix(x[1:(first_na-1),1]), as.matrix(rep(NA,last_na-first_na+1)), 
                           as.matrix(x[first_na:(nrow(df_dest)-(last_na-first_na+1)),1])))
  return(y)
}


### Local Projection IRFs
lp_irf <- function(Y, horizon, dep_var, shk, control_vars, smpl_start, smpl_end, 
                   l_ar = 4, l_shk = 4, l_c = 1, h0x = 0, cholx = 1, alpha = 0.9) {
  ### Estimate impulse responses
  IRF <- matrix(0, nrow = horizon+1, ncol = 3)
  colnames(IRF) <- paste0(dep_var$variable, c("", "_lwr", "_upr"))
  IRF <- as_tibble(IRF)
  for (h in 1:(horizon+1)) {
    # Select and transform data
    y <- select(Y, Date, !!dep_var$variable) %>% 
      transform_df(dep_var, h)
    x <- select(Y, Date, !!control_vars$variable) %>% 
      transform_df(control_vars, 1)
    z <- select(Y, Date, !!shk)
    yx <- left_join(y,x, by="Date", suffix = c("", "_ar")) %>% 
      left_join(., z, by="Date")
    
    # Keep only selected sample
    yx <- filter(yx, Date>=smpl_start & Date<=smpl_end)
    
    ### Specify equation
    # Shock
    lp_eq <- paste0(dep_var$tf_variable, " ~ L(", shk, ", ", h-1+h0x, ":", h-1+h0x+l_shk-1, ")")
    for (n in 1:nrow(control_vars)) {
      # AR terms
      if (control_names[n]==dep_var$variable) {
        lp_eq <- paste0(lp_eq, " + L(", control_vars$tf_variable[n], "_ar, ", h-1+cholx, ":", 
                        h-1+cholx+l_ar-1, ")")
      } else { # Other controls
        lp_eq <- paste0(lp_eq, " + L(", control_vars$tf_variable[n], ", ", h-1+cholx, ":", 
                        h+l_c-1+cholx-1, ")")
      }
    }
    
    # Convert to zoo-object
    yx.zoo <- select(yx, -Date)
    yx.zoo <- zoo(yx.zoo, yx$Date)
    lph <- dynlm(as.formula(lp_eq), data = yx.zoo)
    coef <- summary(lph)$coefficients
    if (h>1) {
      coef <- unclass(coeftest(lph, vcov. = NeweyWest(lph, prewhite = F)))
    }
    # Extract coefficient on monetary policy shock
    IRF[h,1] <- coef[min(which(str_detect(rownames(coef), shk))), 1]
    IRF[h,2] <- IRF[h,1] + coef[min(which(str_detect(rownames(coef), shk))), 2]*qnorm((1-alpha)/2, 0, 1)
    IRF[h,3] <- IRF[h,1] + coef[min(which(str_detect(rownames(coef), shk))), 2]*qnorm(1-(1-alpha)/2, 0, 1)
  }
  IRF <- as_tibble(cbind(seq(h0x,nrow(IRF)-1),IRF))
  colnames(IRF)[1] <- "horizon"
  
  return(IRF)
}