####################################################################################################
# RDP 2021-03: Financial Conditions and Downside Risk to Economic Activity in Australia
####################################################################################################
# Does the FCI help predict activity and if so at what lag?
#
# Financial conditions measure is the FCI estimated using a DFM
#
# Activity measures include:
# - Household Final Consumption Expenditure (QRAHHFCE)
# - Gross Domestic Product (QRGDP)
# - Unemployment rate (QRAUR)
# - Total employment (QRAET)
# - Business Investment excluding mining (QRNMBI)
#
# Luke Hartigan, 10-03-2021
####################################################################################################

# Clear the workspace
rm(list = ls(all = TRUE))

# Set directories
d_location <- "Data/"
c_location <- "Code/methods/"
g_location <- "Results/graphs/"
r_location <- "Results/csv/"

# Source required functions
source(paste0(c_location, "gar_methods.R")) # Robust CCF std errors & Granger causality test

# Set up a few options
options(digits = 4)

# Make the png plots?
make_plots <- TRUE

# What are we doing?
cat("Do financial conditions help predict economic activity?...\n")

####################################################################################################
# Preliminary stuff
####################################################################################################

# String time series dates
fci_begin_str <- "1976-12-01"   # Quarter difference
macro_begin_str <- "1977-09-01" # Year-ended difference
ts_end_str <- "2020-06-01"      # *** Manually edit this line ***
ts_freq <- 4

# Numerical time series dates
fci_begin <- get_year_quarter(x = fci_begin_str)
ts_end <- get_year_quarter(x = ts_end_str)

# Number of lags to investigate -- h = 1 or h = 4 (i.e. one quarter or one year ahead)
nlag <- c(1L, 4L)
nl <- length(nlag)
data_freq <- c("qtr", "ye")

# Number of lags to use in the Granger causality tests
maxp <- 4L

# Multiple testing p-value adjustment
method <- "bonferroni"

# Cross correlation function options
ccf_lag <- 12L
alpha <- 0.05
crt <- 1.0 - (alpha / 2.0)
zn <- stats::qnorm(p = crt, lower.tail = TRUE)
nse <- (2L * ccf_lag) + 1L
acf_type <- "correlation"

# Plotting options
if (make_plots) {
    # Plotting dimensions (i.e. 4:3)
    width <-  800L
    height <- 600L
}

####################################################################################################
# Read in the data
####################################################################################################

# Load the previously estimated FCI (official version)
fci_file <- "fci_q_1_s_1_p_1_rdp.RData"
load(paste0(d_location, fci_file))

# Adjust fci to match sample size of the macro data
fci <- ts(data = fci, start = fci_begin, frequency = ts_freq)
fci <- window(x = fci, start = fci_begin, end = ts_end)

# Macro data
macro_series <- c("QRAHHFCE", "QRAGDP", "QRAUR", "QRAET", "QRNMBI")
macro_file <- "macro_data.csv"
macro <- read.csv(paste0(d_location, macro_file), header = TRUE, sep = ',')

# Drop the dates column
macro <- as.matrix(macro[,-1L, drop = FALSE])
nseries <- dim(macro)[2L]

# Loop over each macro series...
for (it in seq_len(nseries)) {

    # Pick a macro series
    series <- macro_series[it]
    y <- macro[, which(colnames(macro) == series)]

    # ...loop over each lag length...
    for (jt in seq_len(nl)) {

        # String describing the frequency of the macro series being analysed
        diff_type <- data_freq[jt]

        # Get the correct string and numerical time series dates for the macro series based on 'nlag'
        ts_begin_str <- ifelse(test = (nlag[jt] == 1L), yes = fci_begin_str, no = macro_begin_str)
        ts_begin <- get_year_quarter(x = ts_begin_str)

        # Convert macro series to growth rates or first differences if UR
        if (identical(x = series, y = "QRAUR")) {
            dy <- diff(y, lag = nlag[jt])                 # unemployment rate
        } else {
            dy <- ppt_growth(x = y, nlag = nlag[jt])      # everything else
        }

        # Convert to a 'ts' objects
        dy <- ts(data = dy, start = ts_begin, frequency = ts_freq)

        # Copy fci so we can keep the original series for later reuse
        fci_t <- fci

        # Adjust FCI if using year-ended macro data
        if (nlag[jt] == 4L) {
            fci_t <- window(x = fci, start = start(dy), end = end(dy))
        }

        ################################################################
        # Granger Causality test for activity and financial conditions #
        ################################################################

        # Test names
        test_names <- c(sprintf("FCI-/->%s", toupper(series)), sprintf("%s-/->FCI", toupper(series)))

        # Granger causality results file name
        gc_results_file <- sprintf("gc_%s_%s_%s_%s", tolower(series), "fci", diff_type, method)

        # Results storage
        gcfs <- matrix(data = NA_real_, nrow = maxp, ncol = 2L)

        # Pairwise Granger Causality Test
        for (i in seq_len(maxp)) {
            gcfs[i, ] <- granger_cause(x = dy, y = fci_t, plag = i, intercept = TRUE)$pvalue
        }

        # Apply p-value adjustment for multiple comparisons
        gcfs_adj <- apply(X = gcfs, MARGIN = 2L, FUN = p.adjust, method = method, n = maxp)
        colnames(gcfs_adj) <- test_names
        rownames(gcfs_adj) <- paste0("p = ", seq_len(maxp))

        #################################################
        # Estimate the Cross Correlation Function (CCF) #
        #################################################

        # Cross correlation results file name
        ccf_results_file <- sprintf("ccf_%s_%s_fci_lag_%d", tolower(series), diff_type, ccf_lag)

        # Compute a few useful values
        nobs <- length(dy)

        # Compute ACF for each series
        # NB: we are explicitly calling the 'stats' version of 'acf'
        acf_fci <- as.numeric(x = stats::acf(x = as.numeric(fci_t), lag.max = ccf_lag,
                              type = acf_type, plot = FALSE, demean = TRUE)$acf)
        acf_dy <- as.numeric(x = stats::acf(x = as.numeric(dy), lag.max = ccf_lag,
                             type = acf_type, plot = FALSE, demean = TRUE)$acf)

        # Compute the sample CCF
        sccf <- ccf(x = as.numeric(dy), y = as.numeric(fci_t),
                    lag.max = ccf_lag, type = acf_type,
                    plot = FALSE, demean = TRUE)

        # Compute the sample CCF using pre-whiten data
        pw_ccf <- prewhiten(x = as.numeric(dy), y = as.numeric(fci_t),
                            lag.max = ccf_lag, type = acf_type,
                            plot = FALSE, demean = TRUE)

        # Extract the relevant data for saving to disk
        ccf_data <- sccf$acf
        pw_ccf_data <- pw_ccf$ccf$acf

        # CCF (1 - alpha) per cent confidence interval
        lr_se <- sqrt(ccf_var(x = acf_fci, y = acf_dy, n = nobs))
        iid_se <- sqrt(1.0 / nobs)
        ci_lr_mat <- matrix(data = NA_real_, nrow = nse, ncol = 2L)
        ci_iid_mat <- matrix(data = NA_real_, nrow = nse, ncol = 2L)

        if (make_plots) {

            # Plot the raw data sample CCF
            png(file = paste0(g_location, ccf_results_file, ".png"),  width = width, height = height)
            main_title <- sprintf("%s (%s) & FCI \U2013 Sample CCF", toupper(series), toupper(diff_type))
            ci <- zn * lr_se
            lb <- floor(min(-ci, min(ccf_data)))
            ub <- ceiling(max(ci, max(ccf_data)))
            plot(sccf, ci.col = "red4", ci.type = "white", ylim = c(lb, ub),
                 main = main_title, ylab = "Cross-correlation")
            abline(h = c(-ci, ci), col = "blue4", lty = 2)
            legend("topright", legend = c("iid c.i.", "robust c.i."),
                   lty = c(2, 2), col = c("red4", "blue4"), bty = 'n', horiz = FALSE)
            dev.off()

        }

        ci_lr_mat[, 1L] <- rep(x = -zn * lr_se, times = nse)
        ci_lr_mat[, 2L] <- rep(x = zn * lr_se, times = nse)
        ci_iid_mat[, 1L] <- rep(x = -zn * iid_se, times = nse)
        ci_iid_mat[, 2L] <- rep(x = zn * iid_se, times = nse)

        # Combine sample CCFs and C.I.s and set row and column labels before saving
        ccf_data <- cbind(ccf_data, ci_lr_mat, pw_ccf_data, ci_iid_mat)
        ccf_data_names <- c(sprintf("%s_fci", series),
                            sprintf("%1.2f%%CI-lrd", alpha),
                            sprintf("%1.2f%%CI-lr", 1.0 - alpha),
                            sprintf("pw_%s_fci", series),
                            sprintf("%1.2f%%CI-iid", alpha),
                            sprintf("%1.2f%%CI-iid", 1.0 - alpha))
        rownames(ccf_data) <- sccf$lag
        colnames(ccf_data) <- ccf_data_names

        ###############################
        # Write results to .csv files #
        ###############################

        # Granger casuality test
        write.table(x = gcfs_adj,
                    file = paste0(r_location, gc_results_file, ".csv"),
                    append = FALSE, quote = FALSE, sep = ',', row.names = TRUE, col.names = NA)

        # Cross correlation function
        write.table(x = ccf_data,
                    file = paste0(r_location, ccf_results_file, ".csv"),
                    append = FALSE, quote = FALSE, sep = ',', row.names = TRUE, col.names = NA)

    } # ...end nlags

} # ...end macro_series

if (make_plots) {
    cat(sprintf("All graphs saved to %s\n", g_location))
}

cat(sprintf("All files saved to %s\n", r_location))

# EOF
