####################################################################################################
# RDP 2021-03: Financial Conditions and Downside Risk to Economic Activity in Australia
####################################################################################################
# Construct a Financial Conditions Index (FCI) for Australia to be used in 'Growth-at-Risk' analysis
#
# Log-likelihood Ratio Test -- Does the data support a dynamic factor model?
#
# Luke Hartigan, 10-03-2021
####################################################################################################

# Clear the workspace
rm(list = ls(all = TRUE))

# Set directories
d_location <- "Data/"
c_location <- "Code/methods/"

# Source required functions
source(paste0(c_location, "qmle_dfm_methods.R"))   # DFM with dynamic loadings

# Set up a few options
options(digits = 4)

# What are we doing?
cat("Log Likelihood Test of the Dynamic Factor Model Specification...\n")

####################################################################################################
# Read in the data -- already transformed and conditionally standardised (tfs)
####################################################################################################

# Transformed (tf) and conditionally standardised (s) financial dataset
fci_data_file <- "fci_data_tfs.RData"
load(paste0(d_location, fci_data_file))
y <- paneltfs

# QMLE-DFM estimation options
q <- 1L                 # Number of dynamic factors
s <- c(0L, 1L)          # Number of lags in the loadings filter
p <- 1L                 # Number of lags in the factor VAR
id_opt <- "DFM2"        # 'Named-factor' normalisation method
na_opt <- "exclude"     # Remove NAs
scale_opt <- FALSE      # Already standardised
sign_opt <- FALSE       # Not necessary as using named factor normalisation
max_iter <- 500L
threshold <- 1E-4
check_increased <- TRUE
verbose <- FALSE

# Estimate DFM for unbalanced panel -- restricted
mr <- qmle_dfm(x = y, q = q, s = s[1L], p = p,
               id_opt = id_opt, na_opt = na_opt, scale_opt = scale_opt,
               sign_opt = sign_opt, max_iter = max_iter, threshold = threshold,
               check_increased = check_increased, verbose = verbose)

loglik_r <- mr$loglik
aic_r <- mr$aic

# Estimate DFM for unbalanced panel -- unrestricted
mu <- qmle_dfm(x = y, q = q, s = s[2L], p = p,
               id_opt = id_opt, na_opt = na_opt, scale_opt = scale_opt,
               sign_opt = sign_opt, max_iter = max_iter, threshold = threshold,
               check_increased = check_increased, verbose = verbose)

loglik_u <- mu$loglik
aic_u <- mu$aic

# Number of restrictions
nrs <- dim(y)[2L]

# Log-likelihood test statistic
lrts <- -2.0 * (loglik_r - loglik_u)
pval <- stats::pchisq(q = lrts, df = nrs, lower.tail = FALSE)

cat("Likelihood Test for DFM Specification\n")
cat(sprintf("loglik (r) = %g, loglik (u) = %g, no. restrictions = %d\n",
            loglik_r, loglik_u, nrs))
cat(sprintf("The log-likelihood ratio is %g and the p-value is %g\n",
            lrts, round(pval, 4)))

cat("Information Criterion for DFM Specification\n")
cat(sprintf("AIC (r) = %g, AIC (u) = %g\n", aic_r, aic_u))

# EOF
