####################################################################################################
# RDP 2021-03: Financial Conditions and Downside Risk to Economic Activity in Australia
####################################################################################################
# Can financial conditions help predict downside risk to economic activity?
#
# We use quantile spacing regression (QSP) to get an estimate of the future distribution
# for economic activity conditional on a current measure of financial conditions
#
# We focus on the quarterly change (qtr) with a 1 quarter lag
# and the year-ended change (ye) with a 4 quarter lag
#
# When estimating the conditional distribution we assume a Skewed T Distribution (ST)
#
# Activity measures include:
# - Household Final Consumption Expenditure (QRAHHFCE)
# - Gross Domestic Product (QRGDP)
# - Unemployment rate (QRAUR)
# - Total employment (QRAET)
# - Business Investment excluding mining (QRNMBI)
#
# Luke Hartigan, 10-03-2021
####################################################################################################

# Clear the workspace
rm(list = ls(all = TRUE))

# Set directories
d_location <- "Data/"
c_location <- "Code/methods/"
g_location <- "Results/graphs/"
r_location <- "Results/csv/"

# Load the required libraries
suppressMessages(library("quantreg"))   # rq.fit.fnb()
suppressMessages(library("sgt"))        # skewed T distribution stuff (i.e. d, p, q)
suppressMessages(library("ucminf"))     # derivative-free unconstrained function minimiser

# Source required functions
source(paste0(c_location, "misc_methods.R"))
source(paste0(c_location, "qreg_spacing_methods.R"))     # project-specific functions (stationary bootstrap)
source(paste0(c_location, "gar_methods.R"))              # project-specific functions (source last)

# Set up a few options
options(digits = 4)

# Make the png plots?
make_graphs <- TRUE

# What are we doing?
cat("\'Growth-at-Risk\' Analysis for the Australian Economy (Skewed T Distribution)...\n")

####################################################################################################
# Preliminary stuff
####################################################################################################

# String time series dates for y and x variables
ts_qtr_begin_str <- "1976-12-01"
ts_ye_begin_str <- "1977-09-01"
ts_end_str <- "2020-06-01"      # *** Manually edit this line ***
ts_freq <- 4

# String time series dates for yhat -- shifted forward 'h' time periods
yh_qtr_begin_str <- "1977-03-01"
yh_ye_begin_str <- "1977-12-01"
yh_end_str <- "2020-09-01"      # *** Manually edit this line ***

# Number of lags to investigate -- h = 1 or h = 4 (i.e. one quarter or one year ahead)
nlag <- c(1L, 4L)
nl <- length(nlag)
data_freq <- c("qtr", "ye")

# Quantile spacing regression options
trunc_opt <- TRUE
bl <- 8 # i.e. 2 year average block length in stationary bootstrap
nrep <- 1000L
verbose <- FALSE
seed <- 12041948L
ci_size <- 0.05
multiple <- 3

# tau vector of quantiles to investigate
tau_vec <- c(0.05, 0.25, 0.5, 0.75, 0.95)   # the model will be over-identified
jstar <- 3L                                 # i.e. 0.5 is the central quantile we estimate spacings around
ntau <- length(tau_vec)

# length of xseries in density estimation
length_x <- 500L

# alpha per cent for the sequential density comparison
# and expected short fall / long rise measure
alpha <- ci_size

# graphing options
if (make_graphs) {

    # Plotting dimensions (i.e. 4:3)
    width <-  800L
    height <- 600L

    # Colours to use in plots
    col_vec <- colorRampPalette(c("red", "purple", "blue", "purple", "red"))
    col_palette <- colorRampPalette(c("#eff3ff", "#bdd7e7", "#6baed6","#2171b5"))

}

####################################################################################################
# Read in the data
####################################################################################################

# Load the previously estimated FCI (official version)
fci_file <- "fci_q_1_s_1_p_1_rdp.RData"
load(paste0(d_location, fci_file))

# Adjust fci to match sample size of the macro data
fci_begin <- get_year_quarter(x = ts_qtr_begin_str)
ts_end <- get_year_quarter(x = ts_end_str)
fci <- ts(data = fci, start = fci_begin, frequency = ts_freq)
fci <- window(x = fci, start = fci_begin, end = ts_end)

# Macro data
macro_series <- c("QRAHHFCE", "QRAGDP", "QRAUR", "QRAET", "QRNMBI")
macro_file <- "macro_data.csv"
macro <- read.csv(paste0(d_location, macro_file), header = TRUE, sep = ',')

# Drop the dates column
macro <- as.matrix(macro[ , -1L, drop = FALSE])
nseries <- dim(macro)[2L]

####################################################################################################
# Estimate Growth-at-Risk
####################################################################################################

cat("Starting Growth-at-Risk Analysis...\n\n")

# Loop over each macro series...
for (it in seq_len(nseries)) {

    # Pick a macro series
    series <- macro_series[it]
    y <- macro[, which(colnames(macro) == series)]

    # ...loop over each lag length...
    for (jt in seq_len(nl)) {

        # String describing the frequency of the macro series being analysed
        diff_type <- data_freq[jt]

        # String and numerical time series date for y and x variables
        ts_begin_str <- ifelse(test = (nlag[jt] == 1L), yes = ts_qtr_begin_str, no = ts_ye_begin_str)
        ts_begin <- get_year_quarter(x = ts_begin_str)

        # String and numerical time series and date sequence for row names for y_t+h
        yh_begin_str <- ifelse(test = (nlag[jt] == 1L), yes = yh_qtr_begin_str, no = yh_ye_begin_str)
        yh_begin <- get_year_quarter(x = yh_begin_str)
        yh_seq <- seq(from = as.Date(yh_begin_str), to = as.Date(yh_end_str), by = "quarter")
        nobs <- length(yh_seq)

        # Convert macro series to growth rates or first differences if UR
        if (identical(x = series, y = "QRAUR")) {
            dy <- diff(x = y, lag = nlag[jt])           # unemployment rate
        } else {
            dy <- ppt_growth(x = y, nlag = nlag[jt])    # everything else
        }

        # Convert to a 'ts' objects
        dy <- ts(data = dy, start = ts_begin, frequency = ts_freq)

        # Copy fci so we can keep the original series for later reuse
        fci_t <- fci

        # Adjust fci_t if using year-ended macro data
        if (nlag[jt] == 4L) {
            fci_t <- window(x = fci, start = start(dy), end = end(dy))
        }

        ######################################################
        # Quantile Regression Analysis                       #
        ######################################################

        if (make_graphs) {

            # Plot activity and financial conditions index against each other
            plot_name_1 <- sprintf("%s_vs_fci_%s", series, diff_type)
            png_name_1 <- paste0(g_location, plot_name_1, ".png")
            png(file = png_name_1, width = width, height = height)
            plot(cbind(dy, fci_t), col = c("blue3", "red3"), las = 1,
                 lwd = c(1, 1), lty = c(1, 1),
                 plot.type = "multiple", xlab = "", ylab = "Index",
                 main = sprintf("%s (%s) and Financial Conditions", toupper(series), toupper(diff_type)))
            dev.off()

        }

        # What are we doing?
        cat(sprintf("Currently estimating \'Growth-at-risk\' for series: %s (%s)\n", series, diff_type))

        # Construct the dependent and regressor variables
        dy_t <- dy[(1L + nlag[jt]):nobs]
        xx_t <- cbind(1.0, dy[1L:(nobs - nlag[jt])], fci_t[1L:(nobs - nlag[jt])])
        colnames(xx_t) <- c("const", "dy", "fci")

        # Compute 'small'
        small <- quantile(x = dy_t, probs = 0.01, type = 8)

        # Estimate the Quantile Spacing Regression Model -- full and partial
        qfull <- qreg_spacing(y = dy_t, x = xx_t, alpha = tau_vec, jstar = jstar,
                              trunc_opt = trunc_opt, small = small)
        qpart <- qreg_spacing(y = dy_t, x = xx_t[,-3L], alpha = tau_vec, jstar = jstar, # drop last column
                              trunc_opt = trunc_opt, small = small)

        # Get summary information for each model (i.e. coef, stderr, t-stat, p-values and (1 - alpha%) c.i.)
        rq_fm_results <- summary(object = qfull, bl = bl, nrep = nrep,
                                 verbose = verbose, seed = seed, ci_size = ci_size)$output
        rq_pm_results <- summary(object = qpart, bl = bl, nrep = nrep,
                                 verbose = verbose, seed = seed, ci_size = ci_size)$output

        # Extract the R^{1}(\tau) metric
        r1_fn <- qfull$r1_metric
        r1_pn <- qpart$r1_metric
        r1_results <- rbind(r1_fn, r1_pn)
        rownames(r1_results) <- c("fn","pn")

        # Estimate the fitted quantiles based on the spacing estimates and full sample of regressors
        xx_h <- cbind(1.0, dy, fci_t)
        qf_fit <- predict(object = qfull, newx = xx_h, multiple = multiple)
        qp_fit <- predict(object = qpart, newx = xx_h[,-3L], multiple = multiple) # drop last column

        rownames(qf_fit) <- rownames(qp_fit) <- as.character(yh_seq)
        colnames(qf_fit) <- colnames(qp_fit) <- as.character(tau_vec)

        # Plot series and conditional quantiles
        if (make_graphs) {

            # Convert to 'ts' objects
            qf_fit_t <- ts(data = qf_fit, start = yh_begin, frequency = ts_freq)

            # Plot estimated quantiles series with the actual series
            min_qn <- floor(min(qf_fit_t, dy_t))
            max_qn <- ceiling(max(qf_fit_t, dy_t))

            plot_name_2 <- sprintf("%s_rq_fit_%s", series, diff_type)
            png_name_2 <- paste0(g_location, plot_name_2, ".png")
            png(file = png_name_2, width = width, height = height)
            plot(qf_fit_t[,jstar], type = 'n', col = NA, las = 1,
                 plot.type = "single", ylim = c(min_qn, max_qn),
                 xlab = "", ylab = "Per cent", main = sprintf("%s (%s) \U2013 Conditional Distribution", toupper(series), toupper(diff_type)))
            for (i in seq_len(ntau)) {
                lines(qf_fit_t[,i], col = "red2", lwd = 1, lty = 1)
            }
            abline(h = 0.0, col = "black")
            lines(qf_fit_t[,jstar], col = "blue3", lwd = 1, lty = 1)
            lines(dy, col = "black", lwd = 1)
            dev.off()

        }

        ################################################################################
        # Estimate Skewed T Distribution Parameters and PDFs for each t in t = 1,...,T #
        ################################################################################

        # Compute a few useful items
        init <- c(mean(dy), sd(dy), skewness(dy), 5.0)
        min_x <- floor(min(dy))
        max_x <- ceiling(max(dy))

        fm_cond_st <- seq_st_density(rqfit = qf_fit, tau = tau_vec, init_params = init,
                                     min_x = min_x, max_x = max_x, length.out = length_x)
        pm_cond_st <- seq_st_density(rqfit = qp_fit, tau = tau_vec, init_params = init,
                                     min_x = min_x, max_x = max_x, length.out = length_x)

        # Extract Skewed T distribution parameters
        fm_st_params <- fm_cond_st$st_params
        pm_st_params <- pm_cond_st$st_params
        rownames(fm_st_params) <- rownames(pm_st_params) <- as.character(yh_seq)

        # Extract Skewed T distribution densities
        fm_st_den <- fm_cond_st$st_density
        pm_st_den <- pm_cond_st$st_density
        rownames(fm_st_den) <- rownames(pm_st_den) <- as.character(yh_seq)

        if (make_graphs) {

            # Convert to a 'ts' object
            fm_st_params_t <- ts(data = fm_st_params, start = yh_begin, frequency = ts_freq)

            # Plot the estimated parameters for the Conditional Skewed T distribution
            plot_name_3 <- sprintf("%s_st_params_%s", series, diff_type)
            png_name_3 <- paste0(g_location, plot_name_3, ".png")
            png(file = png_name_3, width = width, height = height)
            def.par <- par(no.readonly = TRUE) # save default, for resetting...
            layout(matrix(data = seq_len(length(init)), nrow = 2L, ncol = 2L, byrow = TRUE))

            for (i in seq_len(length(init))) {
                plot(fm_st_params_t[, i], type = 'l', col = 1, lwd = 1, lty = 1,
                     xlab = "", ylab = "Coefficient", main = "")
                mtext(text = colnames(fm_st_params_t)[i], side = 3, line = -2, outer = FALSE, las = 1)
            }

            par(def.par)  # reset to default
            dev.off()

            # Maximum value to be used in plotting
            nobs <- dim(fm_st_den)[1L]
            max_den <- round(max(fm_st_den, pm_st_den), 1)
            xseries <- seq(from = min_x, to = max_x, length.out = length_x)

            # Plot a sequence of densities -- full model
            plot_name_4 <- sprintf("%s_st_seq_den_fm_%s", series, diff_type)
            png_name_4 <- paste0(g_location, plot_name_4, ".png")
            png(file = png_name_4, width = width, height = height)
            plot(x = xseries, y = fm_st_den[1L, ], type = 'n', col = NA, ylim = c(0.0, max_den), las = 1,
                main = sprintf("%s (%s) Sequence of Densities\nConditional Skewed T Distribution",
                               toupper(series), toupper(diff_type)),
                ylab = "Density", xlab = "Value")
            for (i in seq_len(nobs)) {
                lines(x = xseries, y = fm_st_den[i, ], col = "grey70")
            }
            abline(v = 0.0, col = 1, lty = 2)
            abline(h = 0.0, col = 1)
            dev.off()

            # Plot a sequence of densities -- partial model
            plot_name_5 <- sprintf("%s_st_seq_den_pm_%s", series, diff_type)
            png_name_5 <- paste0(g_location, plot_name_5, ".png")
            png(file = png_name_5, width = width, height = height)
            plot(x = xseries, y = pm_st_den[1L, ], type = 'n', col = NA, ylim = c(0.0, max_den), las = 1,
                main = sprintf("%s (%s) Sequence of Densities\nConditional Skewed T Distribution",
                               toupper(series), toupper(diff_type)),
                ylab = "Density", xlab = "Value")
            for (i in seq_len(nobs)) {
                lines(x = xseries, y = pm_st_den[i, ], col = "grey70")
            }
            abline(v = 0.0, col = 1, lty = 2)
            abline(h = 0.0, col = 1)
            dev.off()

            # Plot a 3D chart of the Conditional Skewed T densities over time
            ts_series <- seq(from = (start(dy)[1L] + start(dy)[2L] / ts_freq),
                             to = (end(dy)[1L] + end(dy)[2L] / ts_freq),
                             length.out = dim(qf_fit)[1L])

            ncolours <- 100L
            tx <- ifelse(test = identical(x = series, y = "QRAUR"), yes = 120, no = 60)
            px <- 40

            # Full model
            plot_name_6 <- sprintf("%s_st_3D_seq_den_fm_%s", series, diff_type)
            png_name_6 <- paste0(g_location, plot_name_6, ".png")
            png(file = png_name_6, width = width, height = height)
            persp(x = ts_series, y = xseries, z = fm_st_den,
                  xlab = "",
                  ylab = "Activity",
                  zlab = "Density",
                  main = sprintf("%s (%s) \U2013 Predicted Densities",
                                 toupper(series), toupper(diff_type)),
                  cex.main = 1.75, font.main = 2,
                  cex.lab = 1.5, font.lab = 1,
                  cex.axis = 1.5, font.axis = 1,
                  theta = tx, phi = px, r = sqrt(3),
                  d = 1, scale = TRUE, expand = 0.3,
                  col = surf_colours(x = fm_st_den, col = col_palette(ncolours)),
                  border = NA, ltheta = tx, lphi = px,
                  shade = 0.2, nticks = 5, ticktype = "detailed")
            dev.off()

            # Partial model
            plot_name_7 <- sprintf("%s_st_3D_seq_den_pm_%s", series, diff_type)
            png_name_7 <- paste0(g_location, plot_name_7, ".png")
            png(file = png_name_7, width = width, height = height)
            persp(x = ts_series, y = xseries, z = pm_st_den,
                  xlab = "",
                  ylab = "Activity",
                  zlab = "Density",
                  main = sprintf("%s (%s) \U2013 Predicted Densities",
                                 toupper(series), toupper(diff_type)),
                  cex.main = 1.75, font.main = 2,
                  cex.lab = 1.5, font.lab = 1,
                  cex.axis = 1.5, font.axis = 1,
                  theta = tx, phi = px, r = sqrt(3),
                  d = 1, scale = TRUE, expand = 0.3,
                  col = surf_colours(x = fm_st_den, col = col_palette(ncolours)),
                  border = NA, ltheta = tx, lphi = px,
                  shade = 0.2, nticks = 5, ticktype = "detailed")
            dev.off()

        }

        #####################################################
        # Expected Short fall / Long rise at alpha per cent #
        #####################################################

        fm_ex_loss <- seq_ex_loss(st_params = fm_st_params, alpha = alpha)
        pm_ex_loss <- seq_ex_loss(st_params = pm_st_params, alpha = alpha)

        colnames(fm_ex_loss) <- colnames(pm_ex_loss) <- paste(c("SF", "LR"), rep(x = alpha, each = 2L), sep = '_')
        rownames(fm_ex_loss) <- rownames(pm_ex_loss) <- as.character(yh_seq)

        if (make_graphs) {

            # Convert to a 'ts' object
            fm_ex_loss_t <- ts(data = fm_ex_loss, start = yh_begin, frequency = ts_freq)
            pm_ex_loss_t <- ts(data = pm_ex_loss, start = yh_begin, frequency = ts_freq)

            # Plot alpha per cent expected short fall / long rise for full and partial models
            min_ex <- min(fm_ex_loss_t, pm_ex_loss_t)
            max_ex <- max(fm_ex_loss_t, pm_ex_loss_t)

            plot_name_8 <- sprintf("%s_st_ex_loss_%s", series, diff_type)
            png_name_8 <- paste0(g_location, plot_name_8, ".png")
            png(file = png_name_8, width = width, height = height)
            plot(fm_ex_loss_t, type = 'l', col = c("red3", "blue3"), las = 1, lwd = c(1, 1),
                 plot.type = "single", ylim = c(min_ex, max_ex),
                 xlab = "", ylab = "",
                 main = sprintf("%s (%s) \U2013 %1.0f%% Expected Shortfall / Longrise",
                                toupper(series), toupper(diff_type), alpha*100.0))
            lines(pm_ex_loss_t[,1L], col = "red", lty = 2)
            lines(pm_ex_loss_t[,2L], col = "blue", lty = 2)
            lines(dy, col = 1)
            abline(h = 0.0, col = 1)
            legend("topright", legend = c("Shortfall", "Longrise"),
                   cex = 1.1, col = c("red3", "blue3"),
                   lwd = c(1, 1), bty = 'n', horiz = FALSE)
            dev.off()

        }

        # What did we do?
        cat(sprintf("Completed \'Growth-at-risk\' for series: %s (%s)\n\n", series, diff_type))

        ###############################
        # Write results to .csv files #
        ###############################

        # Quantile regression summary results -- full model
        write.table(x = rq_fm_results,
                    file = paste0(r_location,
                    sprintf("%s_rq_fm_results_%s", series, diff_type), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

        # Quantile regression summary results -- partial model
        write.table(x = rq_pm_results,
                    file = paste0(r_location,
                    sprintf("%s_rq_pm_results_%s", series, diff_type), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

        # R^{1}(\tau) metric -- full model vs. null model; partial model vs null model
        write.table(x = r1_results,
                    file = paste0(r_location,
                    sprintf("%s_r1_results_%s", series, diff_type), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

        # Quantile regression fitted values -- full model
        write.table(x = qf_fit,
                    file = paste0(r_location,
                    sprintf("%s_qf_fit_%s", series, diff_type), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

        # Quantile regression fitted values -- partial model
        write.table(x = qp_fit,
                    file = paste0(r_location,
                    sprintf("%s_qp_fit_%s", series, diff_type), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

        # Conditional Skewed T Distribution parameters -- full model
        write.table(x = fm_st_params,
                    file = paste0(r_location,
                    sprintf("%s_fm_st_params_%s", series, diff_type), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

        # Conditional Skewed T Distribution parameters -- partial model
        write.table(x = pm_st_params,
                    file = paste0(r_location,
                    sprintf("%s_pm_st_params_%s", series, diff_type), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

        # Conditional Skewed T Distributions -- full model
        write.table(x = fm_st_den,
                    file = paste0(r_location,
                    sprintf("%s_fm_st_den_%s", series, diff_type), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

        # Conditional Skewed T Distributions -- partial model
        write.table(x = pm_st_den,
                    file = paste0(r_location,
                    sprintf("%s_pm_st_den_%s", series, diff_type), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

        # Expected short fall / long rise at alpha per cent -- full model
        write.table(x = fm_ex_loss,
                    file = paste0(r_location,
                    sprintf("%s_fm_ex_loss_%s_alpha_%1.0f", series, diff_type, alpha*100.0), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

        # Expected short fall / long rise at alpha per cent -- partial model
        write.table(x = pm_ex_loss,
                    file = paste0(r_location,
                    sprintf("%s_pm_ex_loss_%s_alpha_%1.0f", series, diff_type, alpha*100.0), ".csv"),
                    append = FALSE, quote = FALSE, sep = ',',
                    row.names = TRUE, col.names = NA)

    } # ...end nlags

} # ...end macro_series

cat("...Completed Growth-at-Risk Analysis.\n")

if (make_graphs) {
    cat(sprintf("All graphs saved to %s\n", g_location))
}

cat(sprintf("All files saved to %s\n", r_location))

# EOF
