####################################################################################################
# RDP 2021-03: Financial Conditions and Downside Risk to Economic Activity in Australia
####################################################################################################
# Construct a Financial Conditions Index (FCI) for Australia to be used in 'Growth-at-Risk' analysis
#
# This file is to determine the estimation options for the DFM that will be uses to construct
# the Financial Conditions Index.
#
# Luke Hartigan, 10-03-2021
####################################################################################################

# Clear the workspace
rm(list = ls(all = TRUE))

# Set directories
d_location <- "Data/"
c_location <- "Code/methods/"
g_location <- "Results/graphs/"

# Source required functions
source(paste0(c_location, "qmle_dfm_methods.R"))   # remove_na_values()
source(paste0(c_location, "gar_methods.R"))        # project-specific functions (source last)

# Set up a few options
options(digits = 4)

# Make the png plots?
make_plots <- TRUE

# Plotting options
if (make_plots) {
    # Plotting dimensions (i.e. 4:3)
    width <-  800L
    height <- 600L
}

# What are we doing?
cat("Determining the DFM estimation options for constructing the FCI...\n")

####################################################################################################
# Read in the data -- already transformed and conditionally standardised (tfs)
####################################################################################################

# String time series dates
ts_begin_str <- "1976-12-01"    # first difference
ts_end_str <- "2020-09-01"      # *** Manually edit this line ***

# Transformed (tf) and conditionally standardised (s) financial dataset
fci_data_file <- "fci_data_tfs.RData"
load(paste0(d_location, fci_data_file))
y <- paneltfs

# Time series dates
ts_begin <- get_year_quarter(ts_begin_str)
ts_freq <- 4
ts_freq_str <- "quarter"
ts_qtr_seq <- seq(from = as.Date(ts_begin_str), to = as.Date(ts_end_str), by = ts_freq_str)

# Convert to a 'ts' object
y <- ts(data = y, start = ts_begin, frequency = ts_freq)

####################################################################################################
# Set up various options
####################################################################################################

# How to handle NAs?
na_opt <- "exclude"     # Remove NAs
r_max <- 10L
scale_opt <- FALSE
max_lag <- 6L

yna <- remove_na_values(x = y, na_opt = na_opt)

# Determine the number of static factors
nsf <- mod_bnic(x = yna, kmax = r_max, scale_opt = scale_opt)
cat("Modified Bai and Ng information criteria\n")
print(nsf)
cat('\n')

# PC Factor estimation options
r <- 2L                 # Number of static factors (based on estimate of q)
norm_opt <- "LN"
scale_opt <- FALSE      # Already standardised
sign_opt <- TRUE
vardec_opt <- TRUE      # We want the explained variation of each factor

# Static factor model
sfm <- pc_factor(x = yna, r = r, norm_opt = norm_opt, scale_opt = scale_opt,
                 sign_opt = sign_opt, vardec_opt = vardec_opt)

fac_pc <- sfm$factors

if (make_plots) {

    # Determine the lag order using ACF/PACF plots
    def.par <- par(no.readonly = TRUE) # save default, for resetting...
    png(file = paste0(g_location, "acf_pacf_sfm.png"),
        width = width, height = height)

    layout(matrix(data = 1L:4L, nrow = 2L, ncol = 2L, byrow = TRUE))

    acf(x = fac_pc[,1L], lag.max = max_lag, main = sprintf("PC-Factor \U2013 %d", 1L))
    pacf(x = fac_pc[,1L], lag.max = max_lag, main = sprintf("PC-Factor \U2013 %d", 1L))

    acf(x = fac_pc[,2L], lag.max = max_lag, main = sprintf("PC-Factor \U2013 %d", 2L))
    pacf(x = fac_pc[,2L], lag.max = max_lag, main = sprintf("PC-Factor \U2013 %d", 2L))

    par(def.par) # ...reset to default

    dev.off()

}

if (make_plots) {
    cat(sprintf("All graphs saved to %s\n", g_location))
}

# EOF
