% Conduct robust Bayesian inference about the impulse responses to a unit
% monetary policy shock under the identifying restrictions in Arias,
% Caldara and Rubio-Ramirez (2019) plus sets of sign restrictions
% on impulse responses imposed up to different horizons.

clear variables
close all
clc

oldFolder = pwd;
addpath('auxFunctions');
cd results
resultsDir = pwd;
cd(oldFolder);

%% Import data.
% Data are obtained from the replication files to Antolin-Diaz and
% Rubio-Ramirez (2018), available on Juan Antolin-Diaz's website:
% https://sites.google.com/site/juanantolindiaz/
% The variables are:
% 1) Real GDP 2) GDP deflator 3) Commodity price index 4) Total reserves
% 5) Nonborrowed reserves 6) Federal funds rate
load('Uhlig_Data_Updated.mat');

% Re-order variables so that the federal funds rate is ordered first (the 
% monetary policy shock is the shock of interest).
data = data(:,[end 1:end-1]);
data(:,2:end) = data(:,2:end)*100;
varNames = {varNames{end}, varNames{1:end-1}};

% Declare any exogenous variables (other than the constant; leave empty 
% if none).
exog = [];

%% Options.
opt.p = 12; % No. of lags in VAR
opt.const = 1; % const = 1 if constant in VAR, = 0 otherwise
opt.ivar = [1 2 3 4 5 6];  % Indices of variables of interest
opt.cumIR = []; % Indices of variables for which cumulative impulse response should be used
opt.H = 60; % Terminal horizon for impulse responses
opt.nonEmpty = 10000; % No. of draws from posterior of phi with non-empty identified set
opt.aalpha = 0.68; % Credibility level for credible intervals
opt.Qdraws = 10000; % Number of draws of Q when approximating identified set
opt.dispIter = 100; % Print number of draws remaining every dispIter draws

rng(23032021); % Set seed for random number generator

%% Identifying restrictions from Arias, Caldara and Rubio-Ramirez (2019).

% Input identifying restrictions.
% Each row of eqRestr contains a vector (i,j,t) representing a 
% particular equality restriction, where t is the type of restriction:
% t = 1: the (ij)th element of A0 is zero
% t = 2: the (ij)th element of A0^(-1) is zero
restr.eqRestr = ...
    [1 5 1; % A0_15 = 0 - fed funds rate does not respond to total reserves
     1 6 1]; % A0_16 = 0 - fed funds rate does not respond to non-borrowed reserves

% Each row of signRestr contains a quintet (i,j,h,s,t) representing a
% particular sign restriction, where t is the type of restriction:
% t = 1: the IR of the ith variable with respect to the jth shock at the 
% hth horizon is nonnegative (s = 1) or nonpositive (s = -1).
% t = 2: the (ij)th element of A0 is nonnegative (s = 1) or nonpositive 
% (s = -1). 
% Warning: signRestr should NOT include sign normalisations on elements of 
% A0. These are imposed by default.
% Functions assume that the first sign restriction restricts the impact
% response of the first variable to the first shock to be nonnegative.
S = ...
   [1 1 0 1 1; % Response of fed funds rate to its own shock is nonnegative
    1 2 0 -1 2; % Response of fed funds rate to output is nonnegative
    1 3 0 -1 2]; % Response of fed funds rate to prices is nonnegative

%% Sign restrictions on impulse responses out to 3 months.
H = 3; % Restrictions on horizons h = 0,...,H-1
S1 = zeros(4*H,5);
for hh = 1:H

    S1(hh,:) = [1, 1, (hh-1), 1, 1]; % Response of ffr to shock after h months is nonnegative
    S1(H+hh,:) = [3, 1, (hh-1), -1, 1]; % Response of gdpdef to shock after h months is nonpositive
    S1(2*H+hh,:) = [4, 1, (hh-1), -1, 1]; % Response of com to shock after h months is nonnegative
    S1(3*H+hh,:) = [6, 1, (hh-1), -1, 1]; % Response of nbr to shock after h months is nonnegative

end
restr.signRestr = [S; S1]; % Combine restrictions
   
% Conduct robust Bayesian inference.
mainfile_q1;

fprintf('\nIdentified set unbounded in %1.3g per cent of draws.\n',mean(unbounded)*100)

% Export results to Excel for graphing.
cd(resultsDir);

% Create table of results for impulse responses.
for ii = 1:length(opt.ivar)
    
    TT1 = table((0:opt.H)',etaUnitMed(:,ii),etaUnitCredlb(:,ii),etaUnitCredub(:,ii),...
        etaUnitMedlb(:,ii),etaUnitMedub(:,ii),etaUnitRClb(:,ii),etaUnitRCub(:,ii));
    TT1.Properties.VariableNames = {'Horizon','Median','CREDLB','CREDUB','MEDLB','MEDUB','RCREDLB','RCREDUB'};
    writetable(TT1,'FigureData.xlsx','Sheet',strcat(varNames{ii},'_ACRUhlig3'));
    
end

save('ACRUhlig3_results.mat');

cd(oldFolder);

%% Sign restrictions on impulse responses out to 12 months.
H = 12; % Restrictions on horizons h = 0,...,H-1
S1 = zeros(4*H,5);
for hh = 1:H

    S1(hh,:) = [1, 1, (hh-1), 1, 1]; % Response of ffr to shock after h months is nonnegative
    S1(H+hh,:) = [3, 1, (hh-1), -1, 1]; % Response of gdpdef to shock after h months is nonpositive
    S1(2*H+hh,:) = [4, 1, (hh-1), -1, 1]; % Response of com to shock after h months is nonnegative
    S1(3*H+hh,:) = [6, 1, (hh-1), -1, 1]; % Response of nbr to shock after h months is nonnegative

end
restr.signRestr = [S; S1]; % Combine restrictions
   
% Conduct robust Bayesian inference.
mainfile_q1;

fprintf('\nIdentified set unbounded in %1.3g per cent of draws.\n',mean(unbounded)*100)

% Export results to Excel for graphing.
cd(resultsDir);

% Create table of results for impulse responses.
for ii = 1:length(opt.ivar)
    
    TT1 = table((0:opt.H)',etaUnitMed(:,ii),etaUnitCredlb(:,ii),etaUnitCredub(:,ii),...
        etaUnitMedlb(:,ii),etaUnitMedub(:,ii),etaUnitRClb(:,ii),etaUnitRCub(:,ii));
    TT1.Properties.VariableNames = {'Horizon','Median','CREDLB','CREDUB','MEDLB','MEDUB','RCREDLB','RCREDUB'};
    writetable(TT1,'FigureData.xlsx','Sheet',strcat(varNames{ii},'_ACRUhlig12'));
    
end

save('ACRUhlig12_results.mat');

cd(oldFolder);

%% Sign restrictions on impulse responses out to 24 months.
H = 24; % Restrictions on horizons h = 0,...,H-1
S1 = zeros(4*H,5);
for hh = 1:H

    S1(hh,:) = [1, 1, (hh-1), 1, 1]; % Response of ffr to shock after h months is nonnegative
    S1(H+hh,:) = [3, 1, (hh-1), -1, 1]; % Response of gdpdef to shock after h months is nonpositive
    S1(2*H+hh,:) = [4, 1, (hh-1), -1, 1]; % Response of com to shock after h months is nonnegative
    S1(3*H+hh,:) = [6, 1, (hh-1), -1, 1]; % Response of nbr to shock after h months is nonnegative

end
restr.signRestr = [S; S1]; % Combine restrictions
   
% Conduct robust Bayesian inference.
mainfile_q1;

fprintf('\nIdentified set unbounded in %1.3g per cent of draws.\n',mean(unbounded)*100)

% Export results to Excel for graphing.
cd(resultsDir);

% Create table of results for impulse responses.
for ii = 1:length(opt.ivar)
    
    TT1 = table((0:opt.H)',etaUnitMed(:,ii),etaUnitCredlb(:,ii),etaUnitCredub(:,ii),...
        etaUnitMedlb(:,ii),etaUnitMedub(:,ii),etaUnitRClb(:,ii),etaUnitRCub(:,ii));
    TT1.Properties.VariableNames = {'Horizon','Median','CREDLB','CREDUB','MEDLB','MEDUB','RCREDLB','RCREDUB'};
    writetable(TT1,'FigureData.xlsx','Sheet',strcat(varNames{ii},'_ACRUhlig24'));
    
end

save('ACRUhlig24_results.mat');

cd(oldFolder);