function [rmin,rmax,hmin,hmax] = approximateBounds(phi,Sbar,r,K,c0,L,burn,f,opt)
% Approximate bounds of the identified set for different parameters
% via simulation.
% Inputs:
% - phi: structure containing reduced-form VAR parameters
% - Sbar: matrix containing sign restrictions projected into linear
%   subspace satisfying zero restrictions (after change of basis).
% - r: number of zero restrictions
% - K: change of basis matrix
% - c0: initial value satisfying restrictions in transformed basis
% - L: target number of draws
% - burn: drop initial burn draws
% - f: thinning parameter - keep every f-th draw
% - opt: structure containing options

H = opt.H;
ivar = opt.ivar;
Qdraws = opt.Qdraws;
vma = phi.vma;

%% Compute identified sets for responses to standard-deviation shock.
% Obtain draws from space of orthonormal matrices satisfying restrictions
% (using Gibbs sampler).
q0 = drawqGibbs(Sbar,r,K,c0,L,burn,f);

rDraw = zeros(H+1,length(ivar),Qdraws);

for hh = 1:H+1 % For each horizon

    % Extract required rows of horizon-h VMA coefficient matrix.
    Cphi = vma(ivar,:,hh);
    % Multiply by column of Q (for each draw) to obtain impulse responses.
    rDraw(hh,:,:) = Cphi*q0;

end

% Compute minimum and maximum impulse response over draws of Q.
rmin = min(rDraw,[],3);
rmax = max(rDraw,[],3);

%% Compute identified set for impulse responses to unit shock.
% For each draw of Q, divide impulse responses by impact response of first
% variable. Resulting impulse resposes are with respect to a unit shock to
% the first variable.
hDraw = rDraw./rDraw(1,1,:);

% Compute minimum and maximum over draws of Q.
hmin = min(hDraw,[],3);
hmax = max(hDraw,[],3);

end

