## --- Estimate VAR --- 
## Second step of estimation -- feed factors into VAR
## Author: P Hendy

# Set VAR data
factors_ts$Date <- floor_date(factors_ts$Date, unit = "month")
var_data <- left_join(factors_ts, df_aus_stationary, by="Date") %>%
  dplyr::select(Date, everything()) %>% filter(Date <= "2019-12-31")

# Run initial VAR
favar <- vars::VAR(scale(var_data[,-1]), ic="FPE", type="both")

# Impose impact zero restrictions on effect of Australian variables on international factors
restriction_mat <- matrix(c(1, 1, 1, 1, 1, 1, 1, 1, 
                            1, 1, 1, 1, 1, 1, 1, 1, 
                            1, 1, 1, 1, 1, 1, 1, 1, 
                            0, 0, 0, 1, 1, 1, 1, 1,
                            0, 0, 0, 1, 1, 1, 1, 1,
                            0, 0, 0, 1, 1, 1, 1, 1,  
                            0, 0, 0, 1, 1, 1, 1, 1, 
                            0, 0, 0, 1, 1, 1, 1, 1, 
                            1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1),
                            nrow = 8, ncol=10)
restricted_favar <- vars::restrict(favar, method="manual", resmat=restriction_mat)

# Output FEVDs
restricted_fevd <- run_fevd(restricted_favar, n.ahead=16)
restricted_fevd_graph <- rbind(cbind(as.data.frame(restricted_fevd$twi), "TWI") %>% rename(response=`"TWI"`),
                         cbind(as.data.frame(restricted_fevd$cpi), "CPI") %>% rename(response=`"CPI"`),
                         cbind(as.data.frame(restricted_fevd$gdp), "Real GDP") %>% rename(response=`"Real GDP"`),
                         cbind(as.data.frame(restricted_fevd$unemp), "Unemployment") %>% rename(response=`"Unemployment"`),
                         cbind(as.data.frame(restricted_fevd$cash_rate), "Cash rate") %>% rename(response=`"Cash rate"`)) %>%
                    rowwise() %>%
                    mutate(foreign=sum(activity_factor, prices_factor, financial_factor), .keep="unused") %>%
                    dplyr::select(foreign, everything()) %>%
                    cbind(c(1:16, 1:16, 1:16, 1:16, 1:16)) %>%
                    rename('horizon' = 'c(1:16, 1:16, 1:16, 1:16, 1:16)')
restricted_fevd_graph$foreign <- 100*restricted_fevd_graph$foreign
restricted_fevd_graph$experiment <- "baseline"

ggplot(restricted_fevd_graph, aes(x=horizon, y=foreign)) +
  geom_area(fill="lightblue", colour="cadetblue") +
  facet_wrap(~ response) +
  ylim(0,100) +
  ylab("Foreign contribution") +
  xlab("Horizon") +
  ggtitle("Forecast Error Variance Decomposition") +
  theme(strip.text.x = element_text(size = 25))

# FEVD with error bands
restricted_favar_errors <- bootstrap_fevd(x=restricted_favar, n.ahead=16, runs=1000, ci=0.32, seed=14101806, input_resmat=restriction_mat)
restricted_favar_lower <- as.data.frame(restricted_favar_errors$Lower)
colnames(restricted_favar_lower) <- colnames(restricted_fevd[[1]])
restricted_favar_lower <- restricted_favar_lower %>% dplyr::select(-c(activity_factor, prices_factor, financial_factor)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("lower") %>%
  rename('quantile' = `"lower"`) %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))
  
restricted_favar_upper <- as.data.frame(restricted_favar_errors$Upper)
colnames(restricted_favar_upper) <- colnames(restricted_fevd[[1]])
restricted_favar_upper <- restricted_favar_upper %>% dplyr::select(-c(activity_factor, prices_factor, financial_factor)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("lower") %>%
  rename('quantile' = `"lower"`)  %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))

restricted_fevd_graph_errors <- cbind(restricted_fevd_graph, "mean") %>%
  dplyr::select(foreign, response, horizon, `"mean"`) %>%
  rename('quantile' = `"mean"`) 

ggplot() +
  geom_area(restricted_fevd_graph_errors, mapping=aes(x=horizon, y=foreign), fill="lightblue", colour="cadetblue") +
  geom_line(restricted_favar_lower, mapping=aes(x=horizon, y=foreign), linetype="dashed", colour="red") +
  geom_line(restricted_favar_upper, mapping=aes(x=horizon, y=foreign), linetype="dashed", colour="red") +
  facet_wrap(~ response) +
  ylim(0,100) +
  ylab("Foreign contribution") +
  xlab("Horizon") +
  ggtitle("Forecast Error Variance Decomposition") +
  theme(strip.text.x = element_text(size = 25))

base_output <- merge(merge(restricted_fevd_graph_errors, restricted_favar_lower, by=c("response", "horizon")), restricted_favar_upper, by=c("response", "horizon")) %>%
  dplyr::select(-c(quantile.x, quantile.y, quantile)) %>%
  rename('mean' = 'foreign.x', 'lower' = 'foreign.y', 'upper' = 'foreign') %>%
  arrange(response, horizon)

# Graph 5
write.csv(base_output, file=paste0(data_output_folder, "/Graph 5.csv"))

## Set small scale VAR data
df_small_scale$Date <- floor_date(df_small_scale$Date, unit="month")
df_small_scale <- df_small_scale %>% filter(Date<="2019-12-31")
small_var_data <- left_join(df_small_scale, df_aus_stationary, by="Date")

# Run initial small scale VAR
small_var <- vars::VAR(small_var_data[,-1], ic="SC")

# Set impact zero restrictions
small_restriction_mat <- matrix(c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
                                  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
                                  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                  0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1,
                                  0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 
                                  0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 
                                  0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 
                                  0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1,
                                  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1), 
                                nrow=11, ncol=12)
restricted_small_var <- vars::restrict(small_var, method="manual", resmat=small_restriction_mat)

# Small VAR error bars FEVD
small_restricted_fevd <- run_fevd(restricted_small_var, n.ahead=16)
small_restricted_fevd_graph <- rbind(cbind(as.data.frame(small_restricted_fevd$twi), "TWI") %>% rename(response=`"TWI"`),
                                     cbind(as.data.frame(small_restricted_fevd$cpi), "CPI") %>% rename(response=`"CPI"`),
                                     cbind(as.data.frame(small_restricted_fevd$gdp), "Real GDP") %>% rename(response=`"Real GDP"`),
                                     cbind(as.data.frame(small_restricted_fevd$unemp), "Unemployment") %>% rename(response=`"Unemployment"`),
                                     cbind(as.data.frame(small_restricted_fevd$cash_rate), "Cash rate") %>% rename(response=`"Cash rate"`)) %>%
  rowwise() %>%
  mutate(foreign=sum(WTISPLC, US_NGDP_R_SA_XDC, US_PCPI_IX, SPASTT01USM661N, US_FPOLM_PA, IRLTLT01USM156N), .keep="unused") %>%
  dplyr::select(foreign, everything()) %>%
  cbind(c(1:16, 1:16, 1:16, 1:16, 1:16)) %>%
  rename('horizon' = 'c(1:16, 1:16, 1:16, 1:16, 1:16)')
small_restricted_fevd_graph$foreign <- 100*small_restricted_fevd_graph$foreign
small_restricted_fevd_graph$experiment <- "baseline"

small_favar_errors <- bootstrap_fevd(x=restricted_small_var, n.ahead=16, runs=1000, ci=0.32, seed=14101806, input_resmat=small_restriction_mat, foreign_indices=6)
small_favar_lower <- as.data.frame(small_favar_errors$Lower)
colnames(small_favar_lower) <- colnames(small_restricted_fevd[[1]])
small_favar_lower <- small_favar_lower %>% dplyr::select(-c(WTISPLC, US_NGDP_R_SA_XDC, US_PCPI_IX, SPASTT01USM661N, US_FPOLM_PA, IRLTLT01USM156N)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("lower") %>%
  rename('quantile' = `"lower"`) %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))

small_favar_upper <- as.data.frame(small_favar_errors$Upper)
colnames(small_favar_upper) <- colnames(small_restricted_fevd[[1]])
small_favar_upper <- small_favar_upper %>% dplyr::select(-c(WTISPLC, US_NGDP_R_SA_XDC, US_PCPI_IX, SPASTT01USM661N, US_FPOLM_PA, IRLTLT01USM156N)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("upper") %>%
  rename('quantile' = `"upper"`)  %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))

small_favar_mean <- as.data.frame(small_favar_errors$Mean)
colnames(small_favar_mean) <- colnames(small_restricted_fevd[[1]])
small_favar_mean <- small_favar_mean %>% dplyr::select(-c(WTISPLC, US_NGDP_R_SA_XDC, US_PCPI_IX, SPASTT01USM661N, US_FPOLM_PA, IRLTLT01USM156N)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("mean") %>%
  rename('quantile' = `"mean"`)  %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))

small_output <- merge(merge(small_favar_mean, small_favar_lower, by=c("response", "horizon")), small_favar_upper, by=c("response", "horizon")) %>%
  dplyr::select(-c(quantile.x, quantile.y, quantile)) %>%
  rename('mean' = 'foreign.x', 'lower' = 'foreign.y', 'upper' = 'foreign') %>%
  arrange(response, horizon)
small_output <- merge(small_output, base_output, by=c("response", "horizon")) %>%
  rename('base_mean' = 'mean.y', 'base_lower' = 'lower.y', 'base_upper' = 'upper.y') %>%
  dplyr::select(response, horizon, base_mean, base_lower, base_upper, everything()) %>%
  arrange(response, horizon)

# Graph 7
write.csv(small_output, paste0(data_output_folder, "/Graph 12_test.csv"))

### China experiments
# Set VAR data
no_china_factors_ts$Date <- floor_date(no_china_factors_ts$Date, unit = "month")
no_china_var_data <- left_join(no_china_factors_ts, df_aus_stationary, by="Date") %>%
  dplyr::select(Date, everything()) %>% filter(Date <= "2019-12-31")

# Run initial VAR
no_china_favar <- vars::VAR(no_china_var_data %>% dplyr::select(-Date), ic="FPE", type="both")

# Impose impact zero restrictions on effect of Australian variables on international factors
restriction_mat <- matrix(c(1, 1, 1, 1, 1, 1, 1, 1, 
                            1, 1, 1, 1, 1, 1, 1, 1, 
                            1, 1, 1, 1, 1, 1, 1, 1, 
                            0, 0, 0, 1, 1, 1, 1, 1,
                            0, 0, 0, 1, 1, 1, 1, 1,
                            0, 0, 0, 1, 1, 1, 1, 1,  
                            0, 0, 0, 1, 1, 1, 1, 1, 
                            0, 0, 0, 1, 1, 1, 1, 1, 
                            1, 1, 1, 1, 1, 1, 1, 1,
                            1, 1, 1, 1, 1, 1, 1, 1),
                          nrow = 8, ncol=10)
no_china_restricted_favar <- vars::restrict(no_china_favar, method="manual", resmat=restriction_mat)

# Output FEVDs
no_china_restricted_fevd <- run_fevd(no_china_restricted_favar, n.ahead=16)
no_china_restricted_fevd_graph <- rbind(cbind(as.data.frame(no_china_restricted_fevd$twi), "TWI") %>% rename(response=`"TWI"`),
                                        cbind(as.data.frame(no_china_restricted_fevd$cpi), "CPI") %>% rename(response=`"CPI"`),
                                        cbind(as.data.frame(no_china_restricted_fevd$gdp), "Real GDP") %>% rename(response=`"Real GDP"`),
                                        cbind(as.data.frame(no_china_restricted_fevd$unemp), "Unemployment") %>% rename(response=`"Unemployment"`),
                                        cbind(as.data.frame(no_china_restricted_fevd$cash_rate), "Cash rate") %>% rename(response=`"Cash rate"`)) %>%
  rowwise() %>%
  mutate(foreign=sum(activity_factor, prices_factor, financial_factor), .keep="unused") %>%
  dplyr::select(foreign, everything()) %>%
  cbind(c(1:16, 1:16, 1:16, 1:16, 1:16)) %>%
  rename('horizon' = 'c(1:16, 1:16, 1:16, 1:16, 1:16)')
no_china_restricted_fevd_graph$foreign <- 100*no_china_restricted_fevd_graph$foreign
no_china_restricted_fevd_graph$experiment <- "baseline"

ggplot(no_china_restricted_fevd_graph, aes(x=horizon, y=foreign)) +
  geom_area(fill="lightblue", colour="cadetblue") +
  facet_wrap(~ response) +
  ylim(0,100) +
  ylab("Foreign contribution") +
  xlab("Horizon") +
  ggtitle("Forecast Error Variance Decomposition") +
  theme(strip.text.x = element_text(size = 25))

# FEVD with error bands
no_china_restricted_favar_errors <- bootstrap_fevd(x=no_china_restricted_favar, n.ahead=16, runs=1000, ci=0.32, seed=14101806, input_resmat=restriction_mat)
no_china_restricted_favar_lower <- as.data.frame(no_china_restricted_favar_errors$Lower)
colnames(no_china_restricted_favar_lower) <- colnames(no_china_restricted_fevd[[1]])
no_china_restricted_favar_lower <- no_china_restricted_favar_lower %>% dplyr::select(-c(activity_factor, prices_factor, financial_factor)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("lower") %>%
  rename('quantile' = `"lower"`) %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))

no_china_restricted_favar_upper <- as.data.frame(no_china_restricted_favar_errors$Upper)
colnames(no_china_restricted_favar_upper) <- colnames(no_china_restricted_fevd[[1]])
no_china_restricted_favar_upper <- no_china_restricted_favar_upper %>% dplyr::select(-c(activity_factor, prices_factor, financial_factor)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("lower") %>%
  rename('quantile' = `"lower"`)  %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))

no_china_restricted_fevd_graph_errors <- cbind(no_china_restricted_fevd_graph, "mean") %>%
  dplyr::select(foreign, response, horizon, `"mean"`) %>%
  rename('quantile' = `"mean"`) #%>%

no_china_output <- merge(merge(no_china_restricted_fevd_graph_errors, no_china_restricted_favar_lower, by=c("response", "horizon")), no_china_restricted_favar_upper, by=c("response", "horizon")) %>%
  dplyr::select(-c(quantile.x, quantile.y, quantile)) %>%
  rename('no_mean' = 'foreign.x', 'no_lower' = 'foreign.y', 'no_upper' = 'foreign') %>%
  arrange(response, horizon)

### China Experiment 2
w_china$Date <- floor_date(w_china$Date, unit = "month")
w_china_var_data <- left_join(w_china, df_aus_stationary, by="Date") %>%
  dplyr::select(Date, everything()) %>% filter(Date <= "2019-12-31")

# Run initial VAR
w_china_favar <- vars::VAR(w_china_var_data %>% dplyr::select(-Date), ic="FPE", type="both")

# Impose impact zero restrictions on effect of Australian variables on international factors
w_china_restriction_mat <- matrix(c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                                    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                                    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                                    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                                    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                                    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                                    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                                    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1),
                                  nrow = 12, ncol=14)
w_china_restricted_favar <- vars::restrict(w_china_favar, method="manual", resmat=w_china_restriction_mat)

# Output FEVDs
w_china_restricted_fevd <- run_fevd(w_china_restricted_favar, n.ahead=16)
w_china_restricted_fevd_graph <- rbind(cbind(as.data.frame(w_china_restricted_fevd$twi), "TWI") %>% rename(response=`"TWI"`),
                                       cbind(as.data.frame(w_china_restricted_fevd$cpi), "CPI") %>% rename(response=`"CPI"`),
                                       cbind(as.data.frame(w_china_restricted_fevd$gdp), "Real GDP") %>% rename(response=`"Real GDP"`),
                                       cbind(as.data.frame(w_china_restricted_fevd$unemp), "Unemployment") %>% rename(response=`"Unemployment"`),
                                       cbind(as.data.frame(w_china_restricted_fevd$cash_rate), "Cash rate") %>% rename(response=`"Cash rate"`)) %>%
  rowwise() %>%
  mutate(foreign=sum(activity_factor, prices_factor, financial_factor, china_gdp, china_ip, China_10y, CN_PCPI_IX), .keep="unused") %>%
  dplyr::select(foreign, everything()) %>%
  cbind(c(1:16, 1:16, 1:16, 1:16, 1:16)) %>%
  rename('horizon' = 'c(1:16, 1:16, 1:16, 1:16, 1:16)')
w_china_restricted_fevd_graph$foreign <- 100*w_china_restricted_fevd_graph$foreign
w_china_restricted_fevd_graph$experiment <- "baseline"

ggplot(w_china_restricted_fevd_graph, aes(x=horizon, y=foreign)) +
  geom_area(fill="lightblue", colour="cadetblue") +
  facet_wrap(~ response) +
  ylim(0,100) +
  ylab("Foreign contribution") +
  xlab("Horizon") +
  ggtitle("Forecast Error Variance Decomposition") +
  theme(strip.text.x = element_text(size = 25))

# FEVD with error bands
w_china_restricted_favar_errors <- bootstrap_fevd(x=w_china_restricted_favar, n.ahead=16, runs=1000, ci=0.32, seed=14101806, input_resmat=w_china_restriction_mat)
w_china_restricted_favar_lower <- as.data.frame(w_china_restricted_favar_errors$Lower)
colnames(w_china_restricted_favar_lower) <- colnames(w_china_restricted_fevd[[1]])
w_china_restricted_favar_lower <- w_china_restricted_favar_lower %>% dplyr::select(-c(activity_factor, prices_factor, financial_factor, china_gdp, china_ip, China_10y, CN_PCPI_IX)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("lower") %>%
  rename('quantile' = `"lower"`) %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))

w_china_restricted_favar_upper <- as.data.frame(w_china_restricted_favar_errors$Upper)
colnames(w_china_restricted_favar_upper) <- colnames(w_china_restricted_fevd[[1]])
w_china_restricted_favar_upper <- w_china_restricted_favar_upper %>% dplyr::select(-c(activity_factor, prices_factor, financial_factor, china_gdp, china_ip, China_10y, CN_PCPI_IX)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("lower") %>%
  rename('quantile' = `"lower"`)  %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))

w_china_restricted_fevd_graph_errors <- cbind(w_china_restricted_fevd_graph, "mean") %>%
  dplyr::select(foreign, response, horizon, `"mean"`) %>%
  rename('quantile' = `"mean"`) #%>%

w_china_output <- merge(merge(w_china_restricted_fevd_graph_errors, w_china_restricted_favar_lower, by=c("response", "horizon")), w_china_restricted_favar_upper, by=c("response", "horizon")) %>%
  dplyr::select(-c(quantile.x, quantile.y, quantile)) %>%
  rename('w_mean' = 'foreign.x', 'w_lower' = 'foreign.y', 'w_upper' = 'foreign') %>%
  arrange(response, horizon)

### China Experiment 3
only_china <- w_china %>% dplyr::select(-c(activity_factor, prices_factor, financial_factor))
only_china$Date <- floor_date(only_china$Date, unit = "month")
only_china_var_data <- left_join(only_china, df_aus_stationary, by="Date") %>%
  dplyr::select(Date, everything()) %>% filter(Date <= "2019-12-31")

# Run initial VAR
only_china_favar <- vars::VAR(only_china_var_data %>% dplyr::select(-Date), ic="FPE", type="both")

# Impose impact zero restrictions on effect of Australian variables on international factors
only_china_restriction_mat <- matrix(c(1, 1, 1, 1, 1, 1, 1, 1, 1,
                                       1, 1, 1, 1, 1, 1, 1, 1, 1,
                                       1, 1, 1, 1, 1, 1, 1, 1, 1,
                                       1, 1, 1, 1, 1, 1, 1, 1, 1,
                                       0, 0, 0, 0, 1, 1, 1, 1, 1,
                                       0, 0, 0, 0, 1, 1, 1, 1, 1,
                                       0, 0, 0, 0, 1, 1, 1, 1, 1,
                                       0, 0, 0, 0, 1, 1, 1, 1, 1,
                                       0, 0, 0, 0, 1, 1, 1, 1, 1,
                                       1, 1, 1, 1, 1, 1, 1, 1, 1,
                                       1, 1, 1, 1, 1, 1, 1, 1, 1),
                                     nrow = 9, ncol=11)
only_china_restricted_favar <- vars::restrict(only_china_favar, method="manual", resmat=only_china_restriction_mat)

# Output FEVDs
only_china_restricted_fevd <- run_fevd(only_china_restricted_favar, n.ahead=16)
only_china_restricted_fevd_graph <- rbind(cbind(as.data.frame(only_china_restricted_fevd$twi), "TWI") %>% rename(response=`"TWI"`),
                                          cbind(as.data.frame(only_china_restricted_fevd$cpi), "CPI") %>% rename(response=`"CPI"`),
                                          cbind(as.data.frame(only_china_restricted_fevd$gdp), "Real GDP") %>% rename(response=`"Real GDP"`),
                                          cbind(as.data.frame(only_china_restricted_fevd$unemp), "Unemployment") %>% rename(response=`"Unemployment"`),
                                          cbind(as.data.frame(only_china_restricted_fevd$cash_rate), "Cash rate") %>% rename(response=`"Cash rate"`)) %>%
  rowwise() %>%
  mutate(foreign=sum(china_gdp, china_ip, China_10y, CN_PCPI_IX), .keep="unused") %>%
  dplyr::select(foreign, everything()) %>%
  cbind(c(1:16, 1:16, 1:16, 1:16, 1:16)) %>%
  rename('horizon' = 'c(1:16, 1:16, 1:16, 1:16, 1:16)')
only_china_restricted_fevd_graph$foreign <- 100*only_china_restricted_fevd_graph$foreign
only_china_restricted_fevd_graph$experiment <- "baseline"

ggplot(only_china_restricted_fevd_graph, aes(x=horizon, y=foreign)) +
  geom_area(fill="lightblue", colour="cadetblue") +
  facet_wrap(~ response) +
  ylim(0,100) +
  ylab("Foreign contribution") +
  xlab("Horizon") +
  ggtitle("Forecast Error Variance Decomposition") +
  theme(strip.text.x = element_text(size = 25))

# FEVD with error bands
only_china_restricted_favar_errors <- bootstrap_fevd(x=only_china_restricted_favar, n.ahead=16, runs=1000, ci=0.32, seed=14101806, input_resmat=only_china_restriction_mat)
only_china_restricted_favar_lower <- as.data.frame(only_china_restricted_favar_errors$Lower)
colnames(only_china_restricted_favar_lower) <- colnames(only_china_restricted_fevd[[1]])
only_china_restricted_favar_lower <- only_china_restricted_favar_lower %>% dplyr::select(-c(china_gdp, china_ip, China_10y, CN_PCPI_IX)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("lower") %>%
  rename('quantile' = `"lower"`) %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))

only_china_restricted_favar_upper <- as.data.frame(only_china_restricted_favar_errors$Upper)
colnames(only_china_restricted_favar_upper) <- colnames(only_china_restricted_fevd[[1]])
only_china_restricted_favar_upper <- only_china_restricted_favar_upper %>% dplyr::select(-c(china_gdp, china_ip, China_10y, CN_PCPI_IX)) %>%
  cbind(1:16) %>%
  rename('horizon' = '1:16') %>%
  pivot_longer(!horizon, names_to = "response", values_to = "foreign") %>%
  cbind("lower") %>%
  rename('quantile' = `"lower"`)  %>%
  dplyr::select(foreign, response, horizon, quantile) %>%
  mutate(foreign = 100*foreign, response = ifelse(response=="twi", "TWI", ifelse(response=="cpi", "CPI", ifelse(response=="gdp", "Real GDP", ifelse(response=="unemp", "Unemployment", "Cash rate")))))

only_china_restricted_fevd_graph_errors <- cbind(only_china_restricted_fevd_graph, "mean") %>%
  dplyr::select(foreign, response, horizon, `"mean"`) %>%
  rename('quantile' = `"mean"`) #%>%
#rbind(restricted_favar_lower, restricted_favar_upper)

only_china_output <- merge(merge(only_china_restricted_fevd_graph_errors, only_china_restricted_favar_lower, by=c("response", "horizon")), only_china_restricted_favar_upper, by=c("response", "horizon")) %>%
  dplyr::select(-c(quantile.x, quantile.y, quantile)) %>%
  rename('only_mean' = 'foreign.x', 'only_lower' = 'foreign.y', 'only_upper' = 'foreign') %>%
  arrange(response, horizon)

# Combine results and output
china_experiments_output <- cbind(cbind(no_china_output, w_china_output), only_china_output)[,-c(6,7,11,12)]

# Graph 10
write.csv(china_experiments_output, paste0(data_output_folder,"Graph 10.csv"))

## FMG and IRF experiments
# Function to add variables to the VAR one by one
add_fevd_vars <- function(input_var) {
  input_data <- fmg_var_data %>% dplyr::select(activity_factor, prices_factor, financial_factor, twi, gdp, cash_rate, unemp, cpi, input_var) %>% scale()
  added_favar <- vars::VAR(scale(input_data), ic="FPE", type="both")
  added_restriction_mat <- matrix(c(1, 1, 1, 1, 1, 1, 1, 1, 1,
                                    1, 1, 1, 1, 1, 1, 1, 1, 1,
                                    1, 1, 1, 1, 1, 1, 1, 1, 1, 
                                    0, 0, 0, 1, 1, 1, 1, 1, 1,
                                    0, 0, 0, 1, 1, 1, 1, 1, 1,
                                    0, 0, 0, 1, 1, 1, 1, 1, 1,
                                    0, 0, 0, 1, 1, 1, 1, 1, 1,  
                                    0, 0, 0, 1, 1, 1, 1, 1, 1, 
                                    0, 0, 0, 1, 1, 1, 1, 1, 1, 
                                    1, 1, 1, 1, 1, 1, 1, 1, 1,
                                    1, 1, 1, 1, 1, 1, 1, 1, 1),
                                  nrow = 9, ncol=11)
  added_restricted_favar <- vars::restrict(added_favar, method="manual", resmat=added_restriction_mat)
  
  # Error bands
  added_favar_errors <- bootstrap_fevd(x=added_restricted_favar, n.ahead=16, runs=500, ci=0.32, seed=14101806, input_resmat=added_restriction_mat, foreign_indices=3)
  added_favar_errors$Lower <- 100*added_favar_errors$Lower
  added_favar_errors$Upper <- 100*added_favar_errors$Upper
  
  # Output FEVDs
  added_restricted_fevd <- run_fevd(added_restricted_favar, n.ahead=16)
  return_list <- as.data.frame(added_restricted_fevd[[9]]) %>% dplyr::select(activity_factor, prices_factor, financial_factor) %>% rowwise() %>% mutate(foreign = 100*sum(c(activity_factor, prices_factor, financial_factor)), .keep="unused")
  return(cbind(cbind(cbind(return_list, 1:16) %>% rename('Horizon' = '1:16'),added_favar_errors$Lower[,9]),added_favar_errors$Upper[,9]))
}

fmg_factors_ts <- factors_ts
fmg_factors_ts$Date <- floor_date(fmg_factors_ts$Date, unit = "month")
fmg_var_data <- left_join(fmg_factors_ts, df_fmg_stationary, by="Date") %>%
  dplyr::select(Date, everything()) %>% filter(Date <= "2019-12-31") %>%
  head(-2)

# Collect all the FEVDs one by one
extended <- Reduce(function(d1, d2) merge(d1, d2, by = "Horizon", all.x = TRUE, all.y = FALSE), map(colnames(fmg_var_data %>% dplyr::select(-c(Date, activity_factor, prices_factor, financial_factor, twi, gdp, cash_rate, unemp, cpi))), add_fevd_vars))
colnames(extended) <- c("horizon", unlist(lapply(colnames(fmg_var_data %>% dplyr::select(-c(Date, activity_factor, prices_factor, financial_factor, twi, gdp, cash_rate, unemp, cpi))), function(x) c(x, paste0(x,"_lower"), paste0(x, "_upper")))))

# Graph 11
write.csv(extended, paste0(data_output_folder,"/Graph 11.csv"))

## IRFs
extract_varirf <- function(...){
  
  varirf_object <- list(...) #list one or more varirf input objects
  
  get_vec_length <- function(list_item){nrow(list_item[[1]][[1]])}
  
  if (!("varirf" %in% mapply(class, varirf_object))){
    stop("this function only accepts 'varirf' class objects")
  }
  
  if (length(unique(mapply(class, varirf_object)))!=1){
    stop("all input items must be 'varirf' class objects")
  }    
  if (length(unique(mapply(get_vec_length, varirf_object)))!=1){
    stop("all irf vectors must have the same length")   
  }  
  
  period <- as.data.frame(0:(nrow(varirf_object[[1]][[1]][[1]])-1)) 
  names(period) <- "period"
  
  for (l in 1:length(varirf_object)){
    for (i in 1:3){
      for (j in 1:dim(varirf_object[[l]][[i]][[1]])[2]){
        for (k in 1:length(varirf_object[[l]][[1]])){
          temp_colname <- paste(names(varirf_object[[l]][i]), #vector type (irf, lower, or upper)
                                names(varirf_object[[l]][[i]])[k], #impulse name
                                colnames(varirf_object[[l]][[i]][[k]])[j], #response name
                                sep = "_")
          
          temp <- as.data.frame(varirf_object[[l]][[i]][[k]][, j]) #extracts the vector
          
          names(temp) <- temp_colname #add the column name (vectortype_impulse_reponse)
          period <- cbind(period, temp) 
        }
        
      }
    }
  }
  names(period) <- tolower(names(period))
  return(period)
}

## Full FMG version of the VAR for IRFs
fmg_factors_ts <- factors_ts
fmg_factors_ts$Date <- floor_date(fmg_factors_ts$Date, unit = "month")
fmg_var_data <- left_join(fmg_factors_ts, df_fmg_stationary %>% dplyr::select(-c(sprd_bbws_ois3, exports, lending_rates)), by="Date") %>%
  dplyr::select(Date, everything()) %>% filter(Date <= "2019-12-31") %>%
  head(-2)

# Run initial VAR
fmg_favar <- vars::VAR(fmg_var_data[,-1], ic="FPE", type="both")

# Impose impact zero restrictions on effect of Australian variables on international factors
fmg_restriction_mat <- matrix(c(1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
                                1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1),
                              nrow = 14, ncol=16)
fmg_restricted_favar <- vars::restrict(fmg_favar, method="manual", resmat=fmg_restriction_mat)

# IRFs to a shock in the first factor
fmg_factor_irf <- vars::irf(fmg_restricted_favar, impulse="activity_factor", n.ahead=16, cumulative=T, ci=0.68)
fmg_factor_shock_graph <- extract_varirf(fmg_factor_irf) %>% 
  pivot_longer(cols=!period, names_to=c("bound", "variable"), names_pattern="^([^_]*)_?.*?([^_]*_[^_]*)?$", values_to="value") %>%
  rename("horizon" = "period", "stat"="bound", "response"="variable")
fmg_factor_shock_graph$stat[fmg_factor_shock_graph$stat=="irf"] <- "mean"

fmg_factor_shock_graph %>% pivot_wider(names_from = stat, values_from = value) %>% 
  filter(!(response %in% c("activity_factor", "prices_factor", "financial_factor"))) %>%
  ggplot(aes(x = horizon)) + 
  geom_line(aes(y = mean)) +
  geom_ribbon(aes(ymin = lower, ymax = upper),  alpha = 0.2) +
  geom_hline(aes(yintercept = 0)) +
  facet_wrap(~ response, scales = "free_y", ncol = floor(8/3)) +
  labs(title = glue::glue("IRF to shock in Interest Factor")) +
  xlab("horizon") +
  ylab("estimate")+ 
  theme(axis.title.y = element_text(angle = 90, vjust = 0.5) )

irf_outputs <- fmg_factor_shock_graph %>% pivot_wider(names_from = stat, values_from = value) %>% 
  filter(!(response %in% c("activity_factor", "prices_factor", "financial_factor"))) %>%
  arrange(response) %>%
  mutate(mean = ifelse(response %in% c("bank_lending", "factor_deposits", "factor_gdp", "factor_stocks", "factor_twi", "house_prices", "tot_liabilities"),100*mean,mean),lower = ifelse(response %in% c("bank_lending", "factor_deposits", "factor_gdp", "factor_stocks", "factor_twi", "house_prices", "tot_liabilities"),100*lower,lower),upper = ifelse(response %in% c("bank_lending", "factor_deposits", "factor_gdp", "factor_stocks", "factor_twi", "house_prices", "tot_liabilities"),100*upper,upper))

#Graph C3-4
write.csv(irf_outputs, paste0(data_output_folder, "/Graph C3-4.csv"))

# Second factor shock output
fmg_factor_irf <- vars::irf(fmg_restricted_favar, impulse="prices_factor", n.ahead=16, cumulative=T, ci=0.68)
fmg_factor_shock_graph <- extract_varirf(fmg_factor_irf) %>% 
  pivot_longer(cols=!period, names_to=c("bound", "variable"), names_pattern="^([^_]*)_?.*?([^_]*_[^_]*)?$", values_to="value") %>%
  rename("horizon" = "period", "stat"="bound", "response"="variable")
fmg_factor_shock_graph$stat[fmg_factor_shock_graph$stat=="irf"] <- "mean"

fmg_factor_shock_graph %>% pivot_wider(names_from = stat, values_from = value) %>% 
  filter(!(response %in% c("activity_factor", "prices_factor", "financial_factor"))) %>%
  ggplot(aes(x = horizon)) + 
  geom_line(aes(y = mean)) +
  geom_ribbon(aes(ymin = lower, ymax = upper),  alpha = 0.2) +
  geom_hline(aes(yintercept = 0)) +
  facet_wrap(~ response, scales = "free_y", ncol = floor(8/3)) +
  rba_theme(facet = T, x_title = T) +
  labs(title = glue::glue("IRF to shock in Interest Factor")) +
  xlab("horizon") +
  ylab("estimate")+ 
  theme(axis.title.y = element_text(angle = 90, vjust = 0.5) ) +
  rba_title_y_position(y_left_margin = 3)

irf_outputs <- fmg_factor_shock_graph %>% pivot_wider(names_from = stat, values_from = value) %>% 
  filter(!(response %in% c("activity_factor", "prices_factor", "financial_factor"))) %>%
  arrange(response) %>%
  mutate(mean = ifelse(response %in% c("bank_lending", "factor_deposits", "factor_gdp", "factor_stocks", "factor_twi", "house_prices", "tot_liabilities"),100*mean,mean),lower = ifelse(response %in% c("bank_lending", "factor_deposits", "factor_gdp", "factor_stocks", "factor_twi", "house_prices", "tot_liabilities"),100*lower,lower),upper = ifelse(response %in% c("bank_lending", "factor_deposits", "factor_gdp", "factor_stocks", "factor_twi", "house_prices", "tot_liabilities"),100*upper,upper))

# Graph C5-6
write.csv(irf_outputs, paste0(data_output_folder, "/Graph C5-6.csv"))

# Third factor shock output
fmg_factor_irf <- vars::irf(fmg_restricted_favar, impulse="financial_factor", n.ahead=16, cumulative=T, ci=0.68)
fmg_factor_shock_graph <- extract_varirf(fmg_factor_irf) %>% 
  pivot_longer(cols=!period, names_to=c("bound", "variable"), names_pattern="^([^_]*)_?.*?([^_]*_[^_]*)?$", values_to="value") %>%
  rename("horizon" = "period", "stat"="bound", "response"="variable")
fmg_factor_shock_graph$stat[fmg_factor_shock_graph$stat=="irf"] <- "mean"

fmg_factor_shock_graph %>% pivot_wider(names_from = stat, values_from = value) %>% 
  filter(!(response %in% c("activity_factor", "prices_factor", "financial_factor"))) %>%
  ggplot(aes(x = horizon)) + 
  geom_line(aes(y = mean)) +
  geom_ribbon(aes(ymin = lower, ymax = upper),  alpha = 0.2) +
  geom_hline(aes(yintercept = 0)) +
  facet_wrap(~ response, scales = "free_y", ncol = floor(8/3)) +
  rba_theme(facet = T, x_title = T) +
  labs(title = glue::glue("IRF to shock in Interest Factor")) +
  xlab("horizon") +
  ylab("estimate")+ 
  theme(axis.title.y = element_text(angle = 90, vjust = 0.5) ) +
  rba_title_y_position(y_left_margin = 3)

irf_outputs <- fmg_factor_shock_graph %>% pivot_wider(names_from = stat, values_from = value) %>% 
  filter(!(response %in% c("activity_factor", "prices_factor", "financial_factor"))) %>%
  arrange(response) %>%
  mutate(mean = ifelse(response %in% c("bank_lending", "factor_deposits", "factor_gdp", "factor_stocks", "factor_twi", "house_prices", "tot_liabilities"),100*mean,mean),lower = ifelse(response %in% c("bank_lending", "factor_deposits", "factor_gdp", "factor_stocks", "factor_twi", "house_prices", "tot_liabilities"),100*lower,lower),upper = ifelse(response %in% c("bank_lending", "factor_deposits", "factor_gdp", "factor_stocks", "factor_twi", "house_prices", "tot_liabilities"),100*upper,upper))

# Graph 12-C2
write.csv(irf_outputs, paste0(data_output_folder, "/Graph 12-C2.csv"))